import React, { useState, useEffect } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  TextField,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Switch,
  FormControlLabel,
  Box,
  Typography,
  IconButton,
  Alert,
  CircularProgress,
  Autocomplete,
  InputAdornment,
} from '@mui/material';
import {
  Close,
  Save,
  Visibility,
  VisibilityOff,
  Email,
  Person,
  Phone,
  Badge,
} from '@mui/icons-material';
import { UserProfile, UserCreateRequest, UserUpdateRequest, Role } from '../../types';
import { userManagementService } from '../../services';

interface UserDialogProps {
  open: boolean;
  onClose: () => void;
  onSave: (user: UserProfile) => void;
  user?: UserProfile | null;
  roles: Role[];
}

export const UserDialog: React.FC<UserDialogProps> = ({
  open,
  onClose,
  onSave,
  user,
  roles,
}) => {
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');
  const [showPassword, setShowPassword] = useState(false);
  const [formData, setFormData] = useState<{
    email: string;
    name: string;
    phone: string;
    password: string;
    confirmPassword: string;
    role: string;
    employeeId: string;
    department: string;
    position: string;
    isActive: boolean;
    sendWelcomeEmail: boolean;
  }>({
    email: '',
    name: '',
    phone: '',
    password: '',
    confirmPassword: '',
    role: 'employee',
    employeeId: '',
    department: '',
    position: '',
    isActive: true,
    sendWelcomeEmail: true,
  });

  useEffect(() => {
    if (user) {
      setFormData({
        email: user.email,
        name: user.name,
        phone: user.phone || '',
        password: '',
        confirmPassword: '',
        role: user.role,
        employeeId: user.employeeId || '',
        department: user.department || '',
        position: user.position || '',
        isActive: user.isActive,
        sendWelcomeEmail: false,
      });
    } else {
      setFormData({
        email: '',
        name: '',
        phone: '',
        password: '',
        confirmPassword: '',
        role: 'employee',
        employeeId: '',
        department: '',
        position: '',
        isActive: true,
        sendWelcomeEmail: true,
      });
    }
    setError('');
  }, [user, open]);

  const handleInputChange = (field: string, value: any) => {
    setFormData({ ...formData, [field]: value });
  };

  const handleSave = async () => {
    setLoading(true);
    setError('');

    try {
      // Validation
      if (!user && formData.password !== formData.confirmPassword) {
        setError('Passwords do not match');
        setLoading(false);
        return;
      }

      let result: UserProfile | null;

      if (user) {
        // Update user
        const updateData: UserUpdateRequest = {
          name: formData.name,
          role: formData.role as any,
          phone: formData.phone,
          department: formData.department,
          position: formData.position,
          isActive: formData.isActive,
        };
        
        result = await userManagementService.updateUser(user.id, updateData);
      } else {
        // Create user
        const createData: UserCreateRequest = {
          email: formData.email,
          name: formData.name,
          role: formData.role as any,
          phone: formData.phone,
          password: formData.password,
          employeeId: formData.employeeId,
          department: formData.department,
          position: formData.position,
          sendWelcomeEmail: formData.sendWelcomeEmail,
        };

        // Validate user data
        const validation = await userManagementService.validateUser(createData);
        if (!validation.isValid) {
          setError(validation.errors.join(', '));
          setLoading(false);
          return;
        }

        result = await userManagementService.createUser(createData);
      }

      if (result) {
        onSave(result);
        onClose();
      } else {
        setError('Failed to save user');
      }
    } catch (err: any) {
      setError(err.message || 'Failed to save user');
    } finally {
      setLoading(false);
    }
  };

  const isFormValid = () => {
    const isEmailValid = formData.email.includes('@');
    const isNameValid = formData.name.trim().length >= 2;
    const isPasswordValid = user || formData.password.length >= 6;
    const isPasswordMatch = user || formData.password === formData.confirmPassword;
    
    return isEmailValid && isNameValid && isPasswordValid && isPasswordMatch;
  };

  const departmentOptions = [
    'Human Resources',
    'Engineering',
    'Marketing',
    'Sales',
    'Finance',
    'Operations',
    'IT',
  ];

  return (
    <Dialog
      open={open}
      onClose={onClose}
      maxWidth="md"
      fullWidth
      PaperProps={{
        sx: { minHeight: '70vh' }
      }}
    >
      <DialogTitle sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
        <Typography variant="h6">
          {user ? 'Edit User' : 'Create New User'}
        </Typography>
        <IconButton onClick={onClose}>
          <Close />
        </IconButton>
      </DialogTitle>

      <DialogContent>
        {error && (
          <Alert severity="error" sx={{ mb: 2 }}>
            {error}
          </Alert>
        )}

        <Box sx={{ display: 'flex', flexDirection: 'column', gap: 3, pt: 2 }}>
          {/* Basic Information */}
          <Box>
            <Typography variant="subtitle1" gutterBottom fontWeight="medium">
              Basic Information
            </Typography>
            
            <Box sx={{ display: 'flex', flexDirection: 'column', gap: 2 }}>
              <Box sx={{ display: 'flex', gap: 2, flexWrap: 'wrap' }}>
                <Box sx={{ flex: '1 1 300px' }}>
                  <TextField
                    label="Full Name"
                    value={formData.name}
                    onChange={(e) => handleInputChange('name', e.target.value)}
                    fullWidth
                    required
                    InputProps={{
                      startAdornment: (
                        <InputAdornment position="start">
                          <Person />
                        </InputAdornment>
                      ),
                    }}
                  />
                </Box>

                <Box sx={{ flex: '1 1 300px' }}>
                  <TextField
                    label="Email"
                    type="email"
                    value={formData.email}
                    onChange={(e) => handleInputChange('email', e.target.value)}
                    fullWidth
                    required
                    disabled={!!user}
                    InputProps={{
                      startAdornment: (
                        <InputAdornment position="start">
                          <Email />
                        </InputAdornment>
                      ),
                    }}
                  />
                </Box>
              </Box>

              <Box sx={{ display: 'flex', gap: 2, flexWrap: 'wrap' }}>
                <Box sx={{ flex: '1 1 200px' }}>
                  <TextField
                    label="Phone"
                    value={formData.phone}
                    onChange={(e) => handleInputChange('phone', e.target.value)}
                    fullWidth
                    InputProps={{
                      startAdornment: (
                        <InputAdornment position="start">
                          <Phone />
                        </InputAdornment>
                      ),
                    }}
                  />
                </Box>

                <Box sx={{ flex: '1 1 200px' }}>
                  <FormControl fullWidth>
                    <InputLabel>Role</InputLabel>
                    <Select
                      value={formData.role}
                      onChange={(e) => handleInputChange('role', e.target.value)}
                      required
                    >
                      <MenuItem value="admin">Administrator</MenuItem>
                      <MenuItem value="hr">HR Manager</MenuItem>
                      <MenuItem value="employee">Employee</MenuItem>
                    </Select>
                  </FormControl>
                </Box>

                <Box sx={{ flex: '1 1 200px' }}>
                  <TextField
                    label="Employee ID"
                    value={formData.employeeId}
                    onChange={(e) => handleInputChange('employeeId', e.target.value)}
                    fullWidth
                    InputProps={{
                      startAdornment: (
                        <InputAdornment position="start">
                          <Badge />
                        </InputAdornment>
                      ),
                    }}
                  />
                </Box>
              </Box>
            </Box>
          </Box>

          {/* Work Information */}
          <Box>
            <Typography variant="subtitle1" gutterBottom fontWeight="medium">
              Work Information
            </Typography>
            
            <Box sx={{ display: 'flex', gap: 2, flexWrap: 'wrap' }}>
              <Box sx={{ flex: '1 1 250px' }}>
                <Autocomplete
                  options={departmentOptions}
                  value={formData.department}
                  onChange={(_, newValue) => handleInputChange('department', newValue || '')}
                  renderInput={(params) => (
                    <TextField
                      {...params}
                      label="Department"
                      fullWidth
                    />
                  )}
                  freeSolo
                />
              </Box>

              <Box sx={{ flex: '1 1 250px' }}>
                <TextField
                  label="Position"
                  value={formData.position}
                  onChange={(e) => handleInputChange('position', e.target.value)}
                  fullWidth
                />
              </Box>
            </Box>
          </Box>

          {/* Security */}
          {!user && (
            <Box>
              <Typography variant="subtitle1" gutterBottom fontWeight="medium">
                Security
              </Typography>
              
              <Box sx={{ display: 'flex', gap: 2, flexWrap: 'wrap' }}>
                <Box sx={{ flex: '1 1 250px' }}>
                  <TextField
                    label="Password"
                    type={showPassword ? 'text' : 'password'}
                    value={formData.password}
                    onChange={(e) => handleInputChange('password', e.target.value)}
                    fullWidth
                    required
                    InputProps={{
                      endAdornment: (
                        <InputAdornment position="end">
                          <IconButton
                            onClick={() => setShowPassword(!showPassword)}
                            edge="end"
                          >
                            {showPassword ? <VisibilityOff /> : <Visibility />}
                          </IconButton>
                        </InputAdornment>
                      ),
                    }}
                  />
                </Box>

                <Box sx={{ flex: '1 1 250px' }}>
                  <TextField
                    label="Confirm Password"
                    type={showPassword ? 'text' : 'password'}
                    value={formData.confirmPassword}
                    onChange={(e) => handleInputChange('confirmPassword', e.target.value)}
                    fullWidth
                    required
                    error={formData.password !== formData.confirmPassword && formData.confirmPassword !== ''}
                    helperText={
                      formData.password !== formData.confirmPassword && formData.confirmPassword !== '' 
                        ? 'Passwords do not match' 
                        : ''
                    }
                  />
                </Box>
              </Box>
            </Box>
          )}

          {/* Settings */}
          <Box>
            <Typography variant="subtitle1" gutterBottom fontWeight="medium">
              Settings
            </Typography>
            
            <Box sx={{ display: 'flex', flexDirection: 'column', gap: 2 }}>
              <FormControlLabel
                control={
                  <Switch
                    checked={formData.isActive}
                    onChange={(e) => handleInputChange('isActive', e.target.checked)}
                  />
                }
                label="Active User"
              />
              
              {!user && (
                <FormControlLabel
                  control={
                    <Switch
                      checked={formData.sendWelcomeEmail}
                      onChange={(e) => handleInputChange('sendWelcomeEmail', e.target.checked)}
                    />
                  }
                  label="Send Welcome Email"
                />
              )}
            </Box>
          </Box>
        </Box>
      </DialogContent>

      <DialogActions sx={{ p: 3 }}>
        <Button onClick={onClose}>
          Cancel
        </Button>
        <Button
          variant="contained"
          startIcon={loading ? <CircularProgress size={16} /> : <Save />}
          onClick={handleSave}
          disabled={!isFormValid() || loading}
        >
          {loading ? 'Saving...' : user ? 'Update User' : 'Create User'}
        </Button>
      </DialogActions>
    </Dialog>
  );
};