import React, { useState, useEffect } from 'react';
import {
  Box,
  Typography,
  Button,
  TextField,
  Chip,
  Card,
  CardContent,
  Grid,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Avatar,
  IconButton,
  Tabs,
  Tab,
  Badge,
  SpeedDial,
  SpeedDialAction,
  SpeedDialIcon,
  useTheme,
  useMediaQuery,
  Paper,
} from '@mui/material';
import {
  Search,
  CheckCircle,
  Cancel,
  Schedule,
  AccessTime,
  CalendarToday,
  TrendingUp,
  Add,
  Edit,
  GetApp,
} from '@mui/icons-material';
import { Attendance } from '../types';
import { attendanceService } from '../services/attendanceService';
import { format } from 'date-fns';
import { MobileFilters } from '../components/Common/MobileFilters';
import { ResponsiveTable, MobileCard } from '../components/Common/MobileTable';

interface AttendanceDialogProps {
  open: boolean;
  attendance: Attendance | null;
  onClose: () => void;
  onSave: (attendance: Partial<Attendance>) => void;
  mode: 'view' | 'edit' | 'create';
}

const AttendanceDialog: React.FC<AttendanceDialogProps> = ({
  open,
  attendance,
  onClose,
  onSave,
  mode,
}) => {
  const [formData, setFormData] = useState<Partial<Attendance>>({
    employeeId: '',
    date: format(new Date(), 'yyyy-MM-dd'),
    checkIn: '',
    checkOut: '',
    status: 'present',
    workingHours: 8,
    overtime: 0,
    notes: '',
  });
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down('md'));

  useEffect(() => {
    if (attendance) {
      setFormData(attendance);
    } else if (mode === 'create') {
      setFormData({
        employeeId: '',
        date: format(new Date(), 'yyyy-MM-dd'),
        checkIn: '',
        checkOut: '',
        status: 'present',
        workingHours: 8,
        overtime: 0,
        notes: '',
      });
    }
  }, [attendance, mode]);

  const handleSubmit = () => {
    onSave(formData);
  };

  const isReadonly = mode === 'view';

  return (
    <Dialog 
      open={open} 
      onClose={onClose} 
      maxWidth="sm" 
      fullWidth
      fullScreen={isMobile}
      PaperProps={{
        sx: isMobile ? { 
          margin: 0,
          maxHeight: '100%',
          borderRadius: 0,
        } : undefined
      }}
    >
      <DialogTitle sx={{ 
        pb: 1,
        fontSize: { xs: '1.125rem', sm: '1.25rem' },
        display: 'flex',
        alignItems: 'center',
        justifyContent: 'space-between'
      }}>
        <span>
          {mode === 'create' ? 'Add Attendance Record' : 
           mode === 'edit' ? 'Edit Attendance' : 'Attendance Details'}
        </span>
        {isMobile && (
          <IconButton onClick={onClose} size="small">
            <Cancel />
          </IconButton>
        )}
      </DialogTitle>
      <DialogContent sx={{ px: { xs: 2, sm: 3 } }}>
        <Grid container spacing={2} sx={{ mt: 1 }}>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Employee ID"
              value={formData.employeeId || ''}
              onChange={(e) => setFormData({ ...formData, employeeId: e.target.value })}
              disabled={isReadonly}
              required
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Date"
              type="date"
              value={formData.date || ''}
              onChange={(e) => setFormData({ ...formData, date: e.target.value })}
              disabled={isReadonly}
              InputLabelProps={{ shrink: true }}
              required
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Check In"
              type="time"
              value={formData.checkIn || ''}
              onChange={(e) => setFormData({ ...formData, checkIn: e.target.value })}
              disabled={isReadonly}
              InputLabelProps={{ shrink: true }}
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Check Out"
              type="time"
              value={formData.checkOut || ''}
              onChange={(e) => setFormData({ ...formData, checkOut: e.target.value })}
              disabled={isReadonly}
              InputLabelProps={{ shrink: true }}
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <FormControl fullWidth size={isMobile ? "medium" : "small"}>
              <InputLabel>Status</InputLabel>
              <Select
                value={formData.status || 'present'}
                onChange={(e) => setFormData({ ...formData, status: e.target.value as any })}
                disabled={isReadonly}
                label="Status"
              >
                <MenuItem value="present">Present</MenuItem>
                <MenuItem value="absent">Absent</MenuItem>
                <MenuItem value="late">Late</MenuItem>
                <MenuItem value="half-day">Half Day</MenuItem>
              </Select>
            </FormControl>
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Working Hours"
              type="number"
              value={formData.workingHours || ''}
              onChange={(e) => setFormData({ ...formData, workingHours: Number(e.target.value) })}
              disabled={isReadonly}
              inputProps={{ step: 0.5, min: 0, max: 24 }}
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Overtime Hours"
              type="number"
              value={formData.overtime || ''}
              onChange={(e) => setFormData({ ...formData, overtime: Number(e.target.value) })}
              disabled={isReadonly}
              inputProps={{ step: 0.5, min: 0 }}
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          <Grid size={12}>
            <TextField
              fullWidth
              label="Notes"
              multiline
              rows={2}
              value={formData.notes || ''}
              onChange={(e) => setFormData({ ...formData, notes: e.target.value })}
              disabled={isReadonly}
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
        </Grid>
      </DialogContent>
      <DialogActions sx={{ p: { xs: 2, sm: 3 }, gap: 1 }}>
        <Button onClick={onClose} fullWidth={isMobile}>
          Cancel
        </Button>
        {!isReadonly && (
          <Button 
            onClick={handleSubmit} 
            variant="contained"
            fullWidth={isMobile}
          >
            {mode === 'create' ? 'Add Record' : 'Save Changes'}
          </Button>
        )}
      </DialogActions>
    </Dialog>
  );
};

interface TabPanelProps {
  children?: React.ReactNode;
  index: number;
  value: number;
}

const TabPanel: React.FC<TabPanelProps> = ({ children, value, index }) => (
  <div hidden={value !== index}>
    {value === index && <Box sx={{ py: { xs: 2, sm: 3 } }}>{children}</Box>}
  </div>
);

export const MobileAttendance: React.FC = () => {
  const [attendance, setAttendance] = useState<Attendance[]>([]);
  const [filteredAttendance, setFilteredAttendance] = useState<Attendance[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedAttendance, setSelectedAttendance] = useState<Attendance | null>(null);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [dialogMode, setDialogMode] = useState<'view' | 'edit' | 'create'>('create');
  const [tabValue, setTabValue] = useState(0);
  const [selectedDate, setSelectedDate] = useState(format(new Date(), 'yyyy-MM-dd'));
  const [statusFilter, setStatusFilter] = useState('');
  
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down('md'));

  const mockEmployeeNames: { [key: string]: string } = {
    '1': 'John Doe',
    '2': 'Sarah Johnson',
    '3': 'Michael Brown',
    '4': 'Emily Davis',
    '5': 'David Wilson',
  };

  useEffect(() => {
    loadAttendance();
  }, []);

  useEffect(() => {
    filterAttendance();
  }, [attendance, searchTerm, selectedDate, statusFilter]);

  const loadAttendance = async () => {
    try {
      setLoading(true);
      const data = await attendanceService.getAllAttendance();
      setAttendance(data);
    } catch (error) {
      console.error('Failed to load attendance:', error);
    } finally {
      setLoading(false);
    }
  };

  const filterAttendance = () => {
    let filtered = attendance.filter(att => att.date === selectedDate);

    if (searchTerm) {
      filtered = filtered.filter(att =>
        mockEmployeeNames[att.employeeId]?.toLowerCase().includes(searchTerm.toLowerCase()) ||
        att.employeeId.toLowerCase().includes(searchTerm.toLowerCase())
      );
    }

    if (statusFilter) {
      filtered = filtered.filter(att => att.status === statusFilter);
    }

    setFilteredAttendance(filtered);
  };

  const handleView = (attendance: Attendance) => {
    setSelectedAttendance(attendance);
    setDialogMode('view');
    setDialogOpen(true);
  };

  const handleEdit = (attendance: Attendance) => {
    setSelectedAttendance(attendance);
    setDialogMode('edit');
    setDialogOpen(true);
  };

  const handleCreate = () => {
    setSelectedAttendance(null);
    setDialogMode('create');
    setDialogOpen(true);
  };

  const handleSave = async (attendanceData: Partial<Attendance>) => {
    try {
      if (dialogMode === 'create') {
        await attendanceService.createAttendance(attendanceData as Omit<Attendance, 'id'>);
      } else if (dialogMode === 'edit' && selectedAttendance) {
        await attendanceService.updateAttendance(selectedAttendance.id, attendanceData);
      }
      setDialogOpen(false);
      loadAttendance();
    } catch (error) {
      console.error('Failed to save attendance:', error);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'present': return 'success';
      case 'absent': return 'error';
      case 'late': return 'warning';
      case 'half-day': return 'info';
      default: return 'default';
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'present': return <CheckCircle />;
      case 'absent': return <Cancel />;
      case 'late': return <AccessTime />;
      case 'half-day': return <Schedule />;
      default: return <Schedule />;
    }
  };

  const todayStats = {
    total: filteredAttendance.length,
    present: filteredAttendance.filter(att => att.status === 'present').length,
    absent: filteredAttendance.filter(att => att.status === 'absent').length,
    late: filteredAttendance.filter(att => att.status === 'late').length,
    halfDay: filteredAttendance.filter(att => att.status === 'half-day').length,
  };

  const attendanceRate = todayStats.total > 0 
    ? Math.round(((todayStats.present + todayStats.late + todayStats.halfDay) / todayStats.total) * 100)
    : 0;

  const filters = [
    {
      key: 'status',
      label: 'Status',
      type: 'select' as const,
      value: statusFilter,
      options: [
        { value: 'present', label: 'Present' },
        { value: 'absent', label: 'Absent' },
        { value: 'late', label: 'Late' },
        { value: 'half-day', label: 'Half Day' }
      ]
    },
    {
      key: 'date',
      label: 'Date',
      type: 'date' as const,
      value: selectedDate,
      options: []
    }
  ];

  const handleFilterChange = (key: string, value: string) => {
    if (key === 'status') setStatusFilter(value);
    if (key === 'date') setSelectedDate(value);
  };

  const handleClearFilters = () => {
    setSearchTerm('');
    setStatusFilter('');
    setSelectedDate(format(new Date(), 'yyyy-MM-dd'));
  };

  const columns = [
    {
      field: 'employee',
      label: 'Employee',
      render: (value: any, row: Attendance) => (
        <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
          <Avatar sx={{ width: 32, height: 32 }}>
            {mockEmployeeNames[row.employeeId]?.split(' ').map(n => n[0]).join('') || row.employeeId[0]}
          </Avatar>
          <Box>
            <Typography variant="body2" fontWeight="bold">
              {mockEmployeeNames[row.employeeId] || `Employee ${row.employeeId}`}
            </Typography>
            <Typography variant="body2" color="textSecondary">
              ID: {row.employeeId}
            </Typography>
          </Box>
        </Box>
      ),
    },
    { 
      field: 'date', 
      label: 'Date',
      render: (value: string) => format(new Date(value), 'MMM dd, yyyy')
    },
    { field: 'checkIn', label: 'Check In', render: (value: string) => value || '-' },
    { field: 'checkOut', label: 'Check Out', render: (value: string) => value || '-' },
    {
      field: 'status',
      label: 'Status',
      render: (value: string) => (
        <Chip
          icon={getStatusIcon(value)}
          label={value}
          color={getStatusColor(value) as any}
          size="small"
        />
      ),
    },
    { field: 'workingHours', label: 'Hours', render: (value: number) => `${value}h` },
    { field: 'overtime', label: 'Overtime', render: (value: number) => `${value}h` },
    {
      field: 'actions',
      label: 'Actions',
      render: (value: any, row: Attendance) => (
        <IconButton onClick={() => handleEdit(row)} size="small">
          <Edit />
        </IconButton>
      ),
    },
  ];

  const mobileCardConfig = {
    title: (row: Attendance) => mockEmployeeNames[row.employeeId] || `Employee ${row.employeeId}`,
    subtitle: (row: Attendance) => `${format(new Date(row.date), 'MMM dd, yyyy')} • ID: ${row.employeeId}`,
    avatar: (row: Attendance) => mockEmployeeNames[row.employeeId]?.split(' ').map(n => n[0]).join('') || row.employeeId[0],
    status: (row: Attendance) => ({
      label: row.status,
      color: getStatusColor(row.status) as any,
    }),
    mainFields: ['checkIn', 'checkOut', 'workingHours'],
    expandableFields: ['overtime', 'notes'],
    actions: (row: Attendance) => [
      <IconButton key="edit" onClick={() => handleEdit(row)} size="small">
        <Edit fontSize="small" />
      </IconButton>,
    ],
  };

  const speedDialActions = [
    {
      icon: <Add />,
      name: 'Add Record',
      onClick: handleCreate,
    },
    {
      icon: <GetApp />,
      name: 'Export',
      onClick: () => console.log('Export'),
    },
    {
      icon: <CalendarToday />,
      name: 'Calendar View',
      onClick: () => console.log('Calendar View'),
    },
  ];

  const statsCards = [
    { title: 'Total', value: todayStats.total, icon: <TrendingUp />, color: '#1976d2' },
    { title: 'Present', value: todayStats.present, icon: <CheckCircle />, color: '#2e7d32' },
    { title: 'Absent', value: todayStats.absent, icon: <Cancel />, color: '#d32f2f' },
    { title: 'Late', value: todayStats.late, icon: <AccessTime />, color: '#ed6c02' },
    { title: 'Rate', value: `${attendanceRate}%`, icon: <TrendingUp />, color: '#1976d2' },
  ];

  return (
    <Box>
      <Box sx={{ 
        display: 'flex', 
        justifyContent: 'space-between', 
        alignItems: 'center', 
        mb: { xs: 2, sm: 3 },
        flexDirection: { xs: 'column', sm: 'row' },
        gap: { xs: 2, sm: 0 }
      }}>
        <Typography 
          variant={isMobile ? "h5" : "h4"} 
          sx={{ fontWeight: 600 }}
        >
          Attendance Management
        </Typography>
        {!isMobile && (
          <Button
            variant="contained"
            startIcon={<Add />}
            onClick={handleCreate}
            sx={{ minWidth: 140 }}
          >
            Add Record
          </Button>
        )}
      </Box>
      <Box sx={{ borderBottom: 1, borderColor: 'divider', mb: 2 }}>
        <Tabs 
          value={tabValue} 
          onChange={(e, newValue) => setTabValue(newValue)}
          variant={isMobile ? "scrollable" : "standard"}
          scrollButtons={isMobile ? "auto" : false}
          allowScrollButtonsMobile
        >
          <Tab label="Daily View" />
          <Tab label="Weekly View" />
          <Tab label="Monthly Summary" />
        </Tabs>
      </Box>
      <TabPanel value={tabValue} index={0}>
        {/* Daily Stats Cards */}
        <Grid container spacing={2} sx={{ mb: 3 }}>
          {statsCards.map((stat, index) => (
            <Grid
              key={index}
              size={{
                xs: 6,
                sm: 4,
                md: 2.4
              }}>
              <Card>
                <CardContent sx={{ 
                  textAlign: 'center',
                  py: { xs: 1.5, sm: 2 },
                  px: { xs: 1, sm: 2 }
                }}>
                  <Box sx={{ 
                    display: 'flex', 
                    flexDirection: 'column', 
                    alignItems: 'center',
                    gap: { xs: 0.5, sm: 1 }
                  }}>
                    {React.cloneElement(stat.icon, { 
                      sx: { 
                        fontSize: { xs: 20, sm: 24 },
                        color: stat.color 
                      }
                    })}
                    <Typography 
                      variant="h6" 
                      sx={{ 
                        fontSize: { xs: '1rem', sm: '1.25rem' },
                        fontWeight: 600 
                      }}
                    >
                      {stat.value}
                    </Typography>
                    <Typography 
                      variant="body2" 
                      color="textSecondary"
                      sx={{ fontSize: { xs: '0.75rem', sm: '0.875rem' } }}
                    >
                      {stat.title}
                    </Typography>
                  </Box>
                </CardContent>
              </Card>
            </Grid>
          ))}
        </Grid>

        {/* Filters */}
        <MobileFilters
          searchValue={searchTerm}
          onSearchChange={setSearchTerm}
          filters={filters}
          onFilterChange={handleFilterChange}
          onClearFilters={handleClearFilters}
          placeholder="Search employees..."
        />

        {/* Attendance List/Table */}
        <ResponsiveTable
          columns={columns}
          data={filteredAttendance}
          mobileCardConfig={mobileCardConfig}
          onRowClick={isMobile ? handleView : undefined}
        />
      </TabPanel>
      <TabPanel value={tabValue} index={1}>
        <Typography variant="h6" gutterBottom>
          Weekly Attendance Overview
        </Typography>
        <Typography variant="body1" color="textSecondary">
          Weekly view functionality will be implemented here.
        </Typography>
      </TabPanel>
      <TabPanel value={tabValue} index={2}>
        <Typography variant="h6" gutterBottom>
          Monthly Summary
        </Typography>
        <Typography variant="body1" color="textSecondary">
          Monthly summary and analytics will be implemented here.
        </Typography>
      </TabPanel>
      {/* Mobile Speed Dial */}
      {isMobile && (
        <SpeedDial
          ariaLabel="Attendance actions"
          sx={{ position: 'fixed', bottom: 24, right: 24 }}
          icon={<SpeedDialIcon />}
        >
          {speedDialActions.map((action) => (
            <SpeedDialAction
              key={action.name}
              icon={action.icon}
              tooltipTitle={action.name}
              onClick={action.onClick}
            />
          ))}
        </SpeedDial>
      )}
      <AttendanceDialog
        open={dialogOpen}
        attendance={selectedAttendance}
        onClose={() => setDialogOpen(false)}
        onSave={handleSave}
        mode={dialogMode}
      />
    </Box>
  );
};