import {
  HRPolicy,
  PolicyCreateRequest,
  PolicyUpdateRequest,
  PolicyFilter,
  PolicyListResponse,
  PolicyAcknowledgment,
  ComplianceRule,
  ComplianceRuleCreateRequest,
  ComplianceRuleUpdateRequest,
  ComplianceRecord,
  ComplianceFilter,
  ComplianceListResponse,
  ComplianceCalculationRequest,
  ComplianceReport,
  BulkComplianceOperationRequest,
  ComplianceStats,
  PolicyValidationResult,
  ComplianceValidationResult,
  ComplianceAudit
} from '../types';

// Mock data for development
const mockPolicies: HRPolicy[] = [
  {
    id: 'policy-1',
    title: 'Attendance Policy',
    description: 'Guidelines for employee attendance and punctuality',
    category: 'attendance',
    version: '1.0',
    content: `
      <h2>Attendance Policy</h2>
      <p>All employees are expected to maintain regular attendance and punctuality.</p>
      <h3>Working Hours</h3>
      <ul>
        <li>Standard working hours: 9:00 AM to 6:00 PM</li>
        <li>Lunch break: 1:00 PM to 2:00 PM</li>
        <li>Total working hours per day: 8 hours</li>
      </ul>
      <h3>Late Arrival</h3>
      <p>Employees arriving more than 15 minutes late must inform their supervisor.</p>
      <h3>Absence Reporting</h3>
      <p>Any absence must be reported to HR at least 2 hours in advance.</p>
    `,
    effectiveDate: '2024-01-01',
    status: 'active',
    approvedBy: 'admin@hr.com',
    approvedDate: '2023-12-15',
    createdBy: 'hr@company.com',
    createdAt: '2023-12-01T10:00:00Z',
    updatedAt: '2023-12-15T15:30:00Z',
    tags: ['attendance', 'working-hours', 'punctuality'],
    acknowledgmentRequired: true,
    acknowledgmentCount: 45,
    applicableTo: 'all'
  },
  {
    id: 'policy-2',
    title: 'Leave Management Policy',
    description: 'Comprehensive leave policy including sick leave, annual leave, and special leave',
    category: 'leave',
    version: '2.1',
    content: `
      <h2>Leave Management Policy</h2>
      <h3>Types of Leave</h3>
      <ul>
        <li>Annual Leave: 21 days per year</li>
        <li>Sick Leave: 12 days per year</li>
        <li>Casual Leave: 7 days per year</li>
        <li>Maternity Leave: 26 weeks</li>
        <li>Paternity Leave: 15 days</li>
      </ul>
      <h3>Leave Application Process</h3>
      <p>All leave requests must be submitted through the HR portal at least 3 days in advance.</p>
    `,
    effectiveDate: '2024-01-01',
    expiryDate: '2024-12-31',
    status: 'active',
    approvedBy: 'admin@hr.com',
    approvedDate: '2023-11-20',
    createdBy: 'hr@company.com',
    createdAt: '2023-11-01T09:00:00Z',
    updatedAt: '2023-11-20T14:15:00Z',
    tags: ['leave', 'vacation', 'sick-leave'],
    acknowledgmentRequired: true,
    acknowledgmentCount: 38,
    applicableTo: 'all'
  },
  {
    id: 'policy-3',
    title: 'Code of Conduct',
    description: 'Professional behavior and ethical guidelines for all employees',
    category: 'code_of_conduct',
    version: '1.5',
    content: `
      <h2>Code of Conduct</h2>
      <h3>Professional Behavior</h3>
      <p>All employees must maintain the highest standards of professional conduct.</p>
      <h3>Confidentiality</h3>
      <p>Employees must maintain strict confidentiality of company and client information.</p>
      <h3>Conflict of Interest</h3>
      <p>Employees must disclose any potential conflicts of interest.</p>
    `,
    effectiveDate: '2024-01-01',
    status: 'active',
    approvedBy: 'admin@hr.com',
    approvedDate: '2023-12-10',
    createdBy: 'legal@company.com',
    createdAt: '2023-11-15T11:00:00Z',
    updatedAt: '2023-12-10T16:45:00Z',
    tags: ['conduct', 'ethics', 'confidentiality'],
    acknowledgmentRequired: true,
    acknowledgmentCount: 42,
    applicableTo: 'all'
  }
];

const mockComplianceRules: ComplianceRule[] = [
  {
    id: 'rule-pf-1',
    name: 'Employee Provident Fund (EPF)',
    type: 'pf',
    description: 'PF deductions as per EPF Act 1952',
    applicableFrom: '2024-01-01',
    isActive: true,
    configuration: {
      pfEmployeeRate: 12,
      pfEmployerRate: 12,
      pfWageLimit: 15000,
      pfEpsRate: 8.33,
      pfEdliRate: 0.5
    },
    lastUpdated: '2024-01-01T00:00:00Z',
    updatedBy: 'admin@hr.com'
  },
  {
    id: 'rule-esi-1',
    name: 'Employee State Insurance (ESI)',
    type: 'esi',
    description: 'ESI contributions as per ESI Act 1948',
    applicableFrom: '2024-01-01',
    isActive: true,
    configuration: {
      esiEmployeeRate: 0.75,
      esiEmployerRate: 3.25,
      esiWageLimit: 21000
    },
    lastUpdated: '2024-01-01T00:00:00Z',
    updatedBy: 'admin@hr.com'
  },
  {
    id: 'rule-gratuity-1',
    name: 'Gratuity Calculation',
    type: 'gratuity',
    description: 'Gratuity provision as per Payment of Gratuity Act 1972',
    applicableFrom: '2024-01-01',
    isActive: true,
    configuration: {
      gratuityWageLimit: 0, // No limit for gratuity calculation base
      gratuityYearsRequired: 5,
      gratuityRate: 15, // 15 days for each year
      gratuityMaxLimit: 2000000 // 20 Lakhs
    },
    lastUpdated: '2024-01-01T00:00:00Z',
    updatedBy: 'admin@hr.com'
  }
];

const mockComplianceRecords: ComplianceRecord[] = [
  {
    id: 'comp-1',
    employeeId: '1',
    employeeName: 'John Doe',
    month: '2024-02',
    year: 2024,
    salary: 50000,
    pfEmployee: 1800,
    pfEmployer: 1800,
    pfEps: 1249.5,
    pfEdli: 75,
    pfWages: 15000,
    pfUAN: '123456789012',
    esiEmployee: 157.5,
    esiEmployer: 682.5,
    esiWages: 21000,
    esiNumber: '1234567890',
    gratuityProvision: 1923,
    gratuityWages: 50000,
    tdsAmount: 2500,
    tdsSection: '194A',
    ptAmount: 200,
    ptState: 'Maharashtra',
    status: 'verified',
    calculatedAt: '2024-02-28T10:00:00Z',
    calculatedBy: 'system',
    verifiedAt: '2024-02-29T15:30:00Z',
    verifiedBy: 'hr@company.com'
  }
];

class ComplianceService {
  // Policy Management Methods
  async getPolicies(filter: PolicyFilter = {}): Promise<PolicyListResponse> {
    await new Promise(resolve => setTimeout(resolve, 800));
    
    let filteredPolicies = [...mockPolicies];
    
    // Apply filters
    if (filter.search) {
      filteredPolicies = filteredPolicies.filter(policy =>
        policy.title.toLowerCase().includes(filter.search!.toLowerCase()) ||
        policy.description.toLowerCase().includes(filter.search!.toLowerCase())
      );
    }
    
    if (filter.category) {
      filteredPolicies = filteredPolicies.filter(policy => policy.category === filter.category);
    }
    
    if (filter.status) {
      filteredPolicies = filteredPolicies.filter(policy => policy.status === filter.status);
    }
    
    // Calculate categories
    const categories = mockPolicies.reduce((acc, policy) => {
      const existing = acc.find(cat => cat.category === policy.category);
      if (existing) {
        existing.count++;
      } else {
        acc.push({ category: policy.category, count: 1 });
      }
      return acc;
    }, [] as Array<{ category: HRPolicy['category']; count: number }>);
    
    // Apply pagination
    const limit = filter.limit || 10;
    const offset = filter.offset || 0;
    const paginatedPolicies = filteredPolicies.slice(offset, offset + limit);
    
    return {
      data: paginatedPolicies,
      pagination: {
        total: filteredPolicies.length,
        page: Math.floor(offset / limit) + 1,
        limit,
        totalPages: Math.ceil(filteredPolicies.length / limit),
        hasNext: offset + limit < filteredPolicies.length,
        hasPrevious: offset > 0
      },
      categories
    };
  }

  async getPolicyById(id: string): Promise<HRPolicy | null> {
    await new Promise(resolve => setTimeout(resolve, 300));
    return mockPolicies.find(policy => policy.id === id) || null;
  }

  async createPolicy(request: PolicyCreateRequest): Promise<HRPolicy> {
    await new Promise(resolve => setTimeout(resolve, 1000));
    
    const newPolicy: HRPolicy = {
      id: `policy-${Date.now()}`,
      title: request.title,
      description: request.description,
      category: request.category,
      content: request.content,
      effectiveDate: request.effectiveDate,
      expiryDate: request.expiryDate,
      acknowledgmentRequired: request.acknowledgmentRequired,
      applicableTo: request.applicableTo,
      applicableIds: request.applicableIds,
      tags: request.tags,
      attachments: request.attachments?.map((file: any) => ({
        id: `attachment-${Date.now()}-${Math.random()}`,
        fileName: file.name || 'file',
        fileUrl: file.url || '#',
        fileSize: file.size || 0,
        uploadedAt: new Date().toISOString(),
        uploadedBy: 'current-user@company.com'
      })),
      version: '1.0',
      status: 'draft',
      createdBy: 'current-user@company.com',
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString(),
      acknowledgmentCount: 0
    };
    
    mockPolicies.push(newPolicy);
    return newPolicy;
  }

  async updatePolicy(id: string, request: PolicyUpdateRequest): Promise<HRPolicy | null> {
    await new Promise(resolve => setTimeout(resolve, 800));
    
    const policyIndex = mockPolicies.findIndex(policy => policy.id === id);
    if (policyIndex === -1) return null;
    
    const updatedPolicy: HRPolicy = {
      ...mockPolicies[policyIndex],
      title: request.title ?? mockPolicies[policyIndex].title,
      description: request.description ?? mockPolicies[policyIndex].description,
      category: request.category ?? mockPolicies[policyIndex].category,
      content: request.content ?? mockPolicies[policyIndex].content,
      effectiveDate: request.effectiveDate ?? mockPolicies[policyIndex].effectiveDate,
      expiryDate: request.expiryDate ?? mockPolicies[policyIndex].expiryDate,
      acknowledgmentRequired: request.acknowledgmentRequired ?? mockPolicies[policyIndex].acknowledgmentRequired,
      applicableTo: request.applicableTo ?? mockPolicies[policyIndex].applicableTo,
      applicableIds: request.applicableIds ?? mockPolicies[policyIndex].applicableIds,
      tags: request.tags ?? mockPolicies[policyIndex].tags,
      attachments: request.attachments?.map((file: any) => ({
        id: `attachment-${Date.now()}-${Math.random()}`,
        fileName: file.name || 'file',
        fileUrl: file.url || '#',
        fileSize: file.size || 0,
        uploadedAt: new Date().toISOString(),
        uploadedBy: 'current-user@company.com'
      })) ?? mockPolicies[policyIndex].attachments,
      updatedAt: new Date().toISOString()
    };
    
    mockPolicies[policyIndex] = updatedPolicy;
    return updatedPolicy;
  }

  async deletePolicy(id: string): Promise<boolean> {
    await new Promise(resolve => setTimeout(resolve, 500));
    
    const policyIndex = mockPolicies.findIndex(policy => policy.id === id);
    if (policyIndex === -1) return false;
    
    mockPolicies.splice(policyIndex, 1);
    return true;
  }

  async acknowledgePolicy(policyId: string, employeeId: string): Promise<PolicyAcknowledgment> {
    await new Promise(resolve => setTimeout(resolve, 400));
    
    const acknowledgment: PolicyAcknowledgment = {
      id: `ack-${Date.now()}`,
      policyId,
      employeeId,
      acknowledgedAt: new Date().toISOString(),
      version: '1.0',
      ipAddress: '192.168.1.1'
    };
    
    return acknowledgment;
  }

  async validatePolicy(request: PolicyCreateRequest): Promise<PolicyValidationResult> {
    await new Promise(resolve => setTimeout(resolve, 300));
    
    const errors: string[] = [];
    const warnings: string[] = [];
    
    if (!request.title || request.title.length < 3) {
      errors.push('Title must be at least 3 characters long');
    }
    
    if (!request.content || request.content.length < 50) {
      errors.push('Policy content must be at least 50 characters long');
    }
    
    if (new Date(request.effectiveDate) < new Date()) {
      warnings.push('Effective date is in the past');
    }
    
    return {
      isValid: errors.length === 0,
      errors,
      warnings
    };
  }

  // Compliance Rules Methods
  async getComplianceRules(): Promise<ComplianceRule[]> {
    await new Promise(resolve => setTimeout(resolve, 500));
    return mockComplianceRules.filter(rule => rule.isActive);
  }

  async getComplianceRuleById(id: string): Promise<ComplianceRule | null> {
    await new Promise(resolve => setTimeout(resolve, 200));
    return mockComplianceRules.find(rule => rule.id === id) || null;
  }

  async createComplianceRule(request: ComplianceRuleCreateRequest): Promise<ComplianceRule> {
    await new Promise(resolve => setTimeout(resolve, 600));
    
    const newRule: ComplianceRule = {
      id: `rule-${Date.now()}`,
      ...request,
      isActive: true,
      lastUpdated: new Date().toISOString(),
      updatedBy: 'current-user@company.com'
    };
    
    mockComplianceRules.push(newRule);
    return newRule;
  }

  async updateComplianceRule(id: string, request: ComplianceRuleUpdateRequest): Promise<ComplianceRule | null> {
    await new Promise(resolve => setTimeout(resolve, 700));
    
    const ruleIndex = mockComplianceRules.findIndex(rule => rule.id === id);
    if (ruleIndex === -1) return null;
    
    const updatedRule = {
      ...mockComplianceRules[ruleIndex],
      ...request,
      lastUpdated: new Date().toISOString(),
      updatedBy: 'current-user@company.com'
    };
    
    mockComplianceRules[ruleIndex] = updatedRule;
    return updatedRule;
  }

  // Compliance Records Methods
  async getComplianceRecords(filter: ComplianceFilter = {}): Promise<ComplianceListResponse> {
    await new Promise(resolve => setTimeout(resolve, 1000));
    
    let filteredRecords = [...mockComplianceRecords];
    
    // Apply filters
    if (filter.employeeId) {
      filteredRecords = filteredRecords.filter(record => record.employeeId === filter.employeeId);
    }
    
    if (filter.month) {
      filteredRecords = filteredRecords.filter(record => record.month === filter.month);
    }
    
    if (filter.year) {
      filteredRecords = filteredRecords.filter(record => record.year === filter.year);
    }
    
    if (filter.status) {
      filteredRecords = filteredRecords.filter(record => record.status === filter.status);
    }
    
    // Calculate summary
    const summary = {
      totalEmployees: filteredRecords.length,
      totalWages: filteredRecords.reduce((sum, record) => sum + record.salary, 0),
      totalPFEmployee: filteredRecords.reduce((sum, record) => sum + record.pfEmployee, 0),
      totalPFEmployer: filteredRecords.reduce((sum, record) => sum + record.pfEmployer, 0),
      totalESIEmployee: filteredRecords.reduce((sum, record) => sum + record.esiEmployee, 0),
      totalESIEmployer: filteredRecords.reduce((sum, record) => sum + record.esiEmployer, 0),
      totalTDS: filteredRecords.reduce((sum, record) => sum + record.tdsAmount, 0),
      totalPT: filteredRecords.reduce((sum, record) => sum + record.ptAmount, 0),
      totalGratuity: filteredRecords.reduce((sum, record) => sum + record.gratuityProvision, 0),
      errors: filteredRecords.filter(record => record.errors && record.errors.length > 0).length,
      warnings: 0
    };
    
    // Apply pagination
    const limit = filter.limit || 10;
    const offset = filter.offset || 0;
    const paginatedRecords = filteredRecords.slice(offset, offset + limit);
    
    return {
      data: paginatedRecords,
      pagination: {
        total: filteredRecords.length,
        page: Math.floor(offset / limit) + 1,
        limit,
        totalPages: Math.ceil(filteredRecords.length / limit),
        hasNext: offset + limit < filteredRecords.length,
        hasPrevious: offset > 0
      },
      summary
    };
  }

  async calculateCompliance(request: ComplianceCalculationRequest): Promise<ComplianceRecord[]> {
    await new Promise(resolve => setTimeout(resolve, 2000));
    
    // Mock calculation - in real implementation, this would calculate based on rules and employee data
    const records: ComplianceRecord[] = [];
    const employeeIds = request.employeeIds || ['1', '2', '3'];
    
    employeeIds.forEach(employeeId => {
      const salary = 50000; // Mock salary
      
      // PF calculations
      const pfWages = Math.min(salary, 15000);
      const pfEmployee = Math.round(pfWages * 0.12);
      const pfEmployer = Math.round(pfWages * 0.12);
      const pfEps = Math.round(pfWages * 0.0833);
      const pfEdli = Math.round(pfWages * 0.005);
      
      // ESI calculations
      const esiWages = Math.min(salary, 21000);
      const esiEmployee = Math.round(esiWages * 0.0075);
      const esiEmployer = Math.round(esiWages * 0.0325);
      
      // TDS calculation (simplified)
      const tdsAmount = salary > 50000 ? Math.round(salary * 0.05) : 0;
      
      // PT calculation (simplified for Maharashtra)
      let ptAmount = 0;
      if (salary <= 5000) ptAmount = 0;
      else if (salary <= 10000) ptAmount = 150;
      else ptAmount = 200;
      
      // Gratuity provision (simplified)
      const gratuityProvision = Math.round((salary * 15) / 365 * 30); // 15 days salary per year
      
      const record: ComplianceRecord = {
        id: `comp-${employeeId}-${request.month}-${request.year}`,
        employeeId,
        employeeName: `Employee ${employeeId}`,
        month: request.month,
        year: request.year,
        salary,
        pfEmployee,
        pfEmployer,
        pfEps,
        pfEdli,
        pfWages,
        pfUAN: `${employeeId.padStart(12, '0')}`,
        esiEmployee,
        esiEmployer,
        esiWages,
        esiNumber: `${employeeId.padStart(10, '0')}`,
        gratuityProvision,
        gratuityWages: salary,
        tdsAmount,
        tdsSection: '194A',
        ptAmount,
        ptState: 'Maharashtra',
        status: 'calculated',
        calculatedAt: new Date().toISOString(),
        calculatedBy: 'current-user@company.com'
      };
      
      records.push(record);
    });
    
    return records;
  }

  async validateCompliance(employeeId: string, month: string, year: number): Promise<ComplianceValidationResult> {
    await new Promise(resolve => setTimeout(resolve, 500));
    
    const errors: string[] = [];
    const warnings: string[] = [];
    
    // Mock validation
    const salary = 50000;
    
    const calculatedAmounts = {
      pfEmployee: Math.round(Math.min(salary, 15000) * 0.12),
      pfEmployer: Math.round(Math.min(salary, 15000) * 0.12),
      esiEmployee: Math.round(Math.min(salary, 21000) * 0.0075),
      esiEmployer: Math.round(Math.min(salary, 21000) * 0.0325),
      tds: salary > 50000 ? Math.round(salary * 0.05) : 0,
      pt: salary > 10000 ? 200 : salary > 5000 ? 150 : 0,
      gratuity: Math.round((salary * 15) / 365 * 30)
    };
    
    if (salary < 1000) {
      errors.push('Salary too low for compliance calculations');
    }
    
    if (salary > 100000) {
      warnings.push('High salary - verify tax calculations');
    }
    
    return {
      isValid: errors.length === 0,
      errors,
      warnings,
      calculatedAmounts
    };
  }

  async getComplianceStats(): Promise<ComplianceStats> {
    await new Promise(resolve => setTimeout(resolve, 800));
    
    const totalWages = mockComplianceRecords.reduce((sum, record) => sum + record.salary, 0);
    
    return {
      totalPolicies: mockPolicies.length,
      activePolicies: mockPolicies.filter(p => p.status === 'active').length,
      draftPolicies: mockPolicies.filter(p => p.status === 'draft').length,
      expiredPolicies: mockPolicies.filter(p => p.status === 'expired').length,
      pendingAcknowledgments: mockPolicies.reduce((sum, p) => sum + (p.acknowledgmentRequired ? 50 - (p.acknowledgmentCount || 0) : 0), 0),
      
      totalComplianceRecords: mockComplianceRecords.length,
      calculatedRecords: mockComplianceRecords.filter(r => r.status === 'calculated').length,
      verifiedRecords: mockComplianceRecords.filter(r => r.status === 'verified').length,
      submittedRecords: mockComplianceRecords.filter(r => r.status === 'submitted').length,
      errorRecords: mockComplianceRecords.filter(r => r.errors && r.errors.length > 0).length,
      
      monthlyTotals: {
        wages: totalWages,
        pfEmployee: mockComplianceRecords.reduce((sum, record) => sum + record.pfEmployee, 0),
        pfEmployer: mockComplianceRecords.reduce((sum, record) => sum + record.pfEmployer, 0),
        esiEmployee: mockComplianceRecords.reduce((sum, record) => sum + record.esiEmployee, 0),
        esiEmployer: mockComplianceRecords.reduce((sum, record) => sum + record.esiEmployer, 0),
        tds: mockComplianceRecords.reduce((sum, record) => sum + record.tdsAmount, 0),
        pt: mockComplianceRecords.reduce((sum, record) => sum + record.ptAmount, 0),
        gratuity: mockComplianceRecords.reduce((sum, record) => sum + record.gratuityProvision, 0)
      },
      
      complianceByType: [
        { type: 'pf', total: mockComplianceRecords.length, amount: mockComplianceRecords.reduce((sum, record) => sum + record.pfEmployee + record.pfEmployer, 0) },
        { type: 'esi', total: mockComplianceRecords.length, amount: mockComplianceRecords.reduce((sum, record) => sum + record.esiEmployee + record.esiEmployer, 0) },
        { type: 'tds', total: mockComplianceRecords.length, amount: mockComplianceRecords.reduce((sum, record) => sum + record.tdsAmount, 0) },
        { type: 'pt', total: mockComplianceRecords.length, amount: mockComplianceRecords.reduce((sum, record) => sum + record.ptAmount, 0) },
        { type: 'gratuity', total: mockComplianceRecords.length, amount: mockComplianceRecords.reduce((sum, record) => sum + record.gratuityProvision, 0) }
      ]
    };
  }

  async generateComplianceReport(type: ComplianceReport['type'], period: { from: string; to: string }): Promise<ComplianceReport> {
    await new Promise(resolve => setTimeout(resolve, 3000));
    
    const report: ComplianceReport = {
      id: `report-${Date.now()}`,
      name: `${type.toUpperCase()} Report - ${period.from} to ${period.to}`,
      type,
      period,
      generatedAt: new Date().toISOString(),
      generatedBy: 'current-user@company.com',
      status: 'completed',
      fileUrl: `/api/reports/compliance/${Date.now()}.pdf`,
      summary: {
        totalEmployees: mockComplianceRecords.length,
        totalWages: mockComplianceRecords.reduce((sum, record) => sum + record.salary, 0),
        totalPFEmployee: mockComplianceRecords.reduce((sum, record) => sum + record.pfEmployee, 0),
        totalPFEmployer: mockComplianceRecords.reduce((sum, record) => sum + record.pfEmployer, 0),
        totalESIEmployee: mockComplianceRecords.reduce((sum, record) => sum + record.esiEmployee, 0),
        totalESIEmployer: mockComplianceRecords.reduce((sum, record) => sum + record.esiEmployer, 0),
        totalTDS: mockComplianceRecords.reduce((sum, record) => sum + record.tdsAmount, 0),
        totalPT: mockComplianceRecords.reduce((sum, record) => sum + record.ptAmount, 0),
        totalGratuity: mockComplianceRecords.reduce((sum, record) => sum + record.gratuityProvision, 0),
        errors: 0,
        warnings: 0
      }
    };
    
    return report;
  }
}

export const complianceService = new ComplianceService();