import { 
  Employee, 
  Department, 
  Attendance, 
  LeaveRequest, 
  PayrollRecord, 
  User,
  DashboardStats,
  Holiday 
} from './index';

import {
  // Employee Service Types
  EmployeeFilter,
  EmployeeListResponse,
  EmployeeCreateRequest,
  EmployeeUpdateRequest,
  EmployeeStats,
  EmployeeBulkOperation,
  EmployeeBulkOperationResult,
  EmployeePaginationOptions,

  // Department Service Types
  DepartmentFilter,
  DepartmentListResponse,
  DepartmentCreateRequest,
  DepartmentUpdateRequest,
  DepartmentStats,
  DepartmentPaginationOptions,

  // Attendance Service Types
  AttendanceFilter,
  AttendanceListResponse,
  AttendanceCreateRequest,
  AttendanceUpdateRequest,
  AttendanceStats,
  AttendancePaginationOptions,
  AttendanceTimesheet,

  // Leave Service Types
  LeaveFilter,
  LeaveListResponse,
  LeaveCreateRequest,
  LeaveUpdateRequest,
  LeaveStats,
  LeaveBalance,
  LeavePaginationOptions,

  // Payroll Service Types
  PayrollFilter,
  PayrollListResponse,
  PayrollCreateRequest,
  PayrollUpdateRequest,
  PayrollStats,
  PayrollPaginationOptions,

  // User Service Types
  UserFilter,
  UserListResponse,
  UserProfile,
  UserCreateRequest,
  UserUpdateRequest,
  UserStats,
  UserPaginationOptions,

  // Reports Service Types
  ReportExportOptions,
  ReportExportResult,
  ReportFilter,
  EmployeeReportData,
  AttendanceReportData,
  LeaveReportData,
  PayrollReportData,
  DepartmentReportData,

  // Notification Service Types
  NotificationBase,
  NotificationCreateRequest,
  NotificationListResponse,
  NotificationFilter,
  NotificationStats,

  // Dashboard Service Types
  DashboardMetrics,
  DashboardFilter,
  ExtendedDashboardStats,

  // Holiday Service Types
  HolidayFilter,
  HolidayListResponse,
  HolidayCreateRequest,
  HolidayUpdateRequest,
  HolidayStats,
  HolidayPaginationOptions,
  HolidayCalendarView,
  HolidayBulkOperation,
  HolidayBulkOperationResult,
  HolidayImportOptions,
  HolidayExportOptions,
  HolidayConflict,

  // Common Types
  ServiceResponse,
  PaginatedResponse
} from './services';

// Employee Service Interface
export interface IEmployeeService {
  // CRUD Operations
  getAllEmployees(filter?: EmployeeFilter, pagination?: EmployeePaginationOptions): Promise<EmployeeListResponse>;
  getEmployeeById(id: string): Promise<Employee | null>;
  getEmployeeByEmployeeId(employeeId: string): Promise<Employee | null>;
  createEmployee(employee: EmployeeCreateRequest): Promise<Employee>;
  updateEmployee(id: string, employee: EmployeeUpdateRequest): Promise<Employee | null>;
  deleteEmployee(id: string): Promise<boolean>;

  // Bulk Operations
  bulkCreateEmployees(employees: EmployeeCreateRequest[]): Promise<EmployeeBulkOperationResult>;
  bulkUpdateEmployees(operation: EmployeeBulkOperation): Promise<EmployeeBulkOperationResult>;
  bulkDeleteEmployees(employeeIds: string[]): Promise<EmployeeBulkOperationResult>;

  // Statistics & Analytics
  getEmployeeStats(filter?: EmployeeFilter): Promise<EmployeeStats>;
  searchEmployees(query: string): Promise<Employee[]>;
  getEmployeesByDepartment(departmentId: string): Promise<Employee[]>;

  // Validation
  validateEmployee(employee: EmployeeCreateRequest): Promise<{ isValid: boolean; errors: string[] }>;
}

// Department Service Interface
export interface IDepartmentService {
  // CRUD Operations
  getAllDepartments(filter?: DepartmentFilter, pagination?: DepartmentPaginationOptions): Promise<DepartmentListResponse>;
  getDepartmentById(id: string): Promise<Department | null>;
  createDepartment(department: DepartmentCreateRequest): Promise<Department>;
  updateDepartment(id: string, department: DepartmentUpdateRequest): Promise<Department | null>;
  deleteDepartment(id: string): Promise<boolean>;

  // Statistics & Analytics
  getDepartmentStats(filter?: DepartmentFilter): Promise<DepartmentStats>;
  getDepartmentEmployees(departmentId: string): Promise<Employee[]>;

  // Validation
  validateDepartment(department: DepartmentCreateRequest): Promise<{ isValid: boolean; errors: string[] }>;
}

// Attendance Service Interface
export interface IAttendanceService {
  // CRUD Operations
  getAllAttendance(filter?: AttendanceFilter, pagination?: AttendancePaginationOptions): Promise<AttendanceListResponse>;
  getAttendanceById(id: string): Promise<Attendance | null>;
  getAttendanceByEmployeeId(employeeId: string, dateRange?: { startDate: string; endDate: string }): Promise<Attendance[]>;
  createAttendance(attendance: AttendanceCreateRequest): Promise<Attendance>;
  updateAttendance(id: string, attendance: AttendanceUpdateRequest): Promise<Attendance | null>;
  deleteAttendance(id: string): Promise<boolean>;

  // Specialized Operations
  checkIn(employeeId: string, timestamp: string, location?: any): Promise<Attendance>;
  checkOut(employeeId: string, timestamp: string): Promise<Attendance | null>;
  getAttendanceTimesheet(employeeId: string, startDate: string, endDate: string): Promise<AttendanceTimesheet>;

  // Statistics & Reports
  getAttendanceStats(filter?: AttendanceFilter): Promise<AttendanceStats>;
  getDailyAttendanceReport(date: string): Promise<any>;
  getMonthlyAttendanceReport(month: string, year: number): Promise<any>;

  // Validation
  validateAttendance(attendance: AttendanceCreateRequest): Promise<{ isValid: boolean; errors: string[] }>;
}

// Leave Service Interface
export interface ILeaveService {
  // CRUD Operations
  getAllLeaves(filter?: LeaveFilter, pagination?: LeavePaginationOptions): Promise<LeaveListResponse>;
  getLeaveById(id: string): Promise<LeaveRequest | null>;
  getLeavesByEmployeeId(employeeId: string): Promise<LeaveRequest[]>;
  createLeaveRequest(leave: LeaveCreateRequest): Promise<LeaveRequest>;
  updateLeave(id: string, leave: LeaveUpdateRequest): Promise<LeaveRequest | null>;
  deleteLeave(id: string): Promise<boolean>;

  // Approval Operations
  approveLeave(id: string, approvedBy: string, comments?: string): Promise<LeaveRequest | null>;
  rejectLeave(id: string, rejectedBy: string, comments?: string): Promise<LeaveRequest | null>;

  // Balance & Statistics
  getLeaveBalance(employeeId: string): Promise<LeaveBalance>;
  getLeaveStats(filter?: LeaveFilter): Promise<LeaveStats>;
  getLeaveCalendar(month: number, year: number): Promise<any>;

  // Validation
  validateLeaveRequest(leave: LeaveCreateRequest): Promise<{ isValid: boolean; errors: string[] }>;
}

// Payroll Service Interface
export interface IPayrollService {
  // CRUD Operations
  getAllPayroll(filter?: PayrollFilter, pagination?: PayrollPaginationOptions): Promise<PayrollListResponse>;
  getPayrollById(id: string): Promise<PayrollRecord | null>;
  getPayrollByEmployee(employeeId: string, month?: string, year?: number): Promise<PayrollRecord[]>;
  createPayroll(payroll: PayrollCreateRequest): Promise<PayrollRecord>;
  updatePayroll(id: string, payroll: PayrollUpdateRequest): Promise<PayrollRecord | null>;
  deletePayroll(id: string): Promise<boolean>;

  // Processing Operations
  processPayroll(id: string): Promise<PayrollRecord | null>;
  processMonthlyPayroll(month: string, year: number, departmentId?: string): Promise<PayrollRecord[]>;
  generatePayslip(payrollId: string): Promise<any>;

  // Statistics
  getPayrollStats(filter?: PayrollFilter): Promise<PayrollStats>;
  getPayrollSummary(month: string, year: number): Promise<any>;

  // Validation
  validatePayroll(payroll: PayrollCreateRequest): Promise<{ isValid: boolean; errors: string[] }>;
}

// User Service Interface
export interface IUserService {
  // CRUD Operations
  getAllUsers(filter?: UserFilter, pagination?: UserPaginationOptions): Promise<UserListResponse>;
  getUserById(id: string): Promise<UserProfile | null>;
  getUserByEmail(email: string): Promise<UserProfile | null>;
  createUser(user: UserCreateRequest): Promise<UserProfile>;
  updateUser(id: string, user: UserUpdateRequest): Promise<UserProfile | null>;
  deleteUser(id: string): Promise<boolean>;

  // Authentication
  authenticate(email: string, password: string): Promise<{ user: UserProfile; token: string } | null>;
  refreshToken(token: string): Promise<string | null>;
  logout(token: string): Promise<boolean>;

  // Profile Management
  updateProfile(userId: string, profile: Partial<UserProfile>): Promise<UserProfile | null>;
  changePassword(userId: string, currentPassword: string, newPassword: string): Promise<boolean>;
  resetPassword(email: string): Promise<boolean>;

  // Statistics
  getUserStats(): Promise<UserStats>;

  // Validation
  validateUser(user: UserCreateRequest): Promise<{ isValid: boolean; errors: string[] }>;
}

// Reports Service Interface
export interface IReportsService {
  // Report Generation
  generateEmployeeReport(filter?: ReportFilter): Promise<EmployeeReportData>;
  generateAttendanceReport(filter?: ReportFilter): Promise<AttendanceReportData>;
  generateLeaveReport(filter?: ReportFilter): Promise<LeaveReportData>;
  generatePayrollReport(filter?: ReportFilter): Promise<PayrollReportData>;
  generateDepartmentReport(filter?: ReportFilter): Promise<DepartmentReportData>;

  // Export Operations
  exportReport(
    reportType: 'employee' | 'attendance' | 'leave' | 'payroll' | 'department',
    format: 'csv' | 'excel' | 'pdf',
    data: any,
    filename?: string
  ): Promise<ReportExportResult>;

  // Analytics
  getReportAnalytics(reportType: string, period: string): Promise<any>;
  getPopularReports(): Promise<any[]>;

  // Templates
  getReportTemplates(): Promise<any[]>;
  saveReportTemplate(template: any): Promise<any>;
}

// Notification Service Interface
export interface INotificationService {
  // CRUD Operations
  getAllNotifications(filter?: NotificationFilter): Promise<NotificationListResponse>;
  getNotificationById(id: string): Promise<NotificationBase | null>;
  createNotification(notification: NotificationCreateRequest): Promise<NotificationBase>;
  updateNotification(id: string, notification: Partial<NotificationBase>): Promise<NotificationBase | null>;
  deleteNotification(id: string): Promise<boolean>;

  // User Operations
  getUserNotifications(userId: string, limit?: number): Promise<NotificationBase[]>;
  markAsRead(notificationId: string): Promise<boolean>;
  markAllAsRead(userId: string): Promise<boolean>;

  // Statistics
  getNotificationStats(userId?: string): Promise<NotificationStats>;

  // System Notifications
  sendSystemNotification(notification: NotificationCreateRequest): Promise<boolean>;
  broadcastNotification(notification: NotificationCreateRequest): Promise<boolean>;
}

// Dashboard Service Interface
export interface IDashboardService {
  // Dashboard Data
  getDashboardStats(filter?: DashboardFilter): Promise<ExtendedDashboardStats>;
  getDashboardMetrics(filter?: DashboardFilter): Promise<DashboardMetrics>;
  getRecentActivity(limit?: number): Promise<any[]>;

  // Widget Data
  getWidgetData(widgetType: string, config: any): Promise<any>;
  
  // Analytics
  getOverviewAnalytics(): Promise<any>;
  getTrendAnalytics(period: string): Promise<any>;
  getComparativeAnalytics(): Promise<any>;

  // Real-time Updates
  subscribeToUpdates(userId: string): Promise<boolean>;
  unsubscribeFromUpdates(userId: string): Promise<boolean>;
}

// Holiday Service Interface
export interface IHolidayService {
  // CRUD Operations
  getAllHolidays(filter?: HolidayFilter, pagination?: HolidayPaginationOptions): Promise<HolidayListResponse>;
  getHolidayById(id: string): Promise<Holiday | null>;
  getHolidaysByDateRange(startDate: string, endDate: string, organizationId?: string): Promise<Holiday[]>;
  createHoliday(holiday: HolidayCreateRequest): Promise<Holiday>;
  updateHoliday(id: string, holiday: HolidayUpdateRequest): Promise<Holiday | null>;
  deleteHoliday(id: string): Promise<boolean>;

  // Calendar Operations
  getHolidayCalendarView(year: number, month: number, organizationId?: string): Promise<HolidayCalendarView>;
  getHolidaysForYear(year: number, organizationId?: string): Promise<Holiday[]>;
  getUpcomingHolidays(limit?: number, organizationId?: string): Promise<Holiday[]>;

  // Bulk Operations
  bulkCreateHolidays(holidays: HolidayCreateRequest[]): Promise<HolidayBulkOperationResult>;
  bulkUpdateHolidays(operation: HolidayBulkOperation): Promise<HolidayBulkOperationResult>;
  bulkDeleteHolidays(holidayIds: string[]): Promise<HolidayBulkOperationResult>;

  // Import/Export
  importHolidays(options: HolidayImportOptions): Promise<HolidayBulkOperationResult>;
  exportHolidays(options: HolidayExportOptions): Promise<any>;

  // Validation & Conflicts
  validateHoliday(holiday: HolidayCreateRequest): Promise<{ isValid: boolean; errors: string[] }>;
  checkHolidayConflicts(holiday: HolidayCreateRequest): Promise<HolidayConflict[]>;

  // Statistics & Analytics
  getHolidayStats(filter?: HolidayFilter): Promise<HolidayStats>;
  getHolidaysByType(type: string, year?: number): Promise<Holiday[]>;

  // Organization & Department specific
  getOrganizationHolidays(organizationId: string, year?: number): Promise<Holiday[]>;
  getDepartmentHolidays(departmentIds: string[], year?: number): Promise<Holiday[]>;
}