import React, { useState, useEffect } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  TextField,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Switch,
  FormControlLabel,
  Box,
  Typography,
  IconButton,
  Alert,
  CircularProgress,
  Chip,
  Autocomplete,
} from '@mui/material';
import {
  Close,
  Save,
  ColorLens,
} from '@mui/icons-material';
import { DatePicker } from '@mui/x-date-pickers/DatePicker';
import { LocalizationProvider } from '@mui/x-date-pickers/LocalizationProvider';
import { AdapterDateFns } from '@mui/x-date-pickers/AdapterDateFns';
import { Holiday, HolidayCreateRequest, HolidayUpdateRequest, HolidayConflict } from '../../types';
import { holidayService } from '../../services';

interface HolidayDialogProps {
  open: boolean;
  onClose: () => void;
  onSave: (holiday: Holiday) => void;
  holiday?: Holiday | null;
  organizationId?: string;
  departments?: Array<{ id: string; name: string }>;
  preselectedDate?: Date | null;
}

const HOLIDAY_TYPES = [
  { value: 'public', label: 'Public Holiday' },
  { value: 'company', label: 'Company Holiday' },
  { value: 'optional', label: 'Optional Holiday' },
  { value: 'religious', label: 'Religious Holiday' },
];

const HOLIDAY_COLORS = [
  '#f44336', '#e91e63', '#9c27b0', '#673ab7',
  '#3f51b5', '#2196f3', '#03a9f4', '#00bcd4',
  '#009688', '#4caf50', '#8bc34a', '#cddc39',
  '#ffeb3b', '#ffc107', '#ff9800', '#ff5722',
];

export const HolidayDialog: React.FC<HolidayDialogProps> = ({
  open,
  onClose,
  onSave,
  holiday,
  organizationId,
  departments = [],
  preselectedDate,
}) => {
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');
  const [conflicts, setConflicts] = useState<HolidayConflict[]>([]);
  const [formData, setFormData] = useState<{
    name: string;
    date: Date;
    description: string;
    type: 'public' | 'company' | 'optional' | 'religious';
    isRecurring: boolean;
    organizationId: string;
    departmentIds: string[];
    color: string;
  }>({
    name: '',
    date: preselectedDate || new Date(),
    description: '',
    type: 'company',
    isRecurring: false,
    organizationId: organizationId || '',
    departmentIds: [],
    color: '#2196f3',
  });

  useEffect(() => {
    if (holiday) {
      setFormData({
        name: holiday.name,
        date: new Date(holiday.date),
        description: holiday.description || '',
        type: holiday.type,
        isRecurring: holiday.isRecurring,
        organizationId: holiday.organizationId || '',
        departmentIds: holiday.departmentIds || [],
        color: holiday.color || '#2196f3',
      });
    } else {
      setFormData({
        name: '',
        date: preselectedDate || new Date(),
        description: '',
        type: 'company',
        isRecurring: false,
        organizationId: organizationId || '',
        departmentIds: [],
        color: '#2196f3',
      });
    }
    setError('');
    setConflicts([]);
  }, [holiday, organizationId, preselectedDate, open]);

  const handleInputChange = (field: string, value: any) => {
    setFormData({ ...formData, [field]: value });
    
    // Clear conflicts when date changes
    if (field === 'date') {
      setConflicts([]);
    }
  };

  const checkConflicts = async () => {
    if (formData.date && formData.name) {
      try {
        const holidayRequest: HolidayCreateRequest = {
          ...formData,
          date: formData.date.toISOString().split('T')[0],
        };
        
        const foundConflicts = await holidayService.checkHolidayConflicts(holidayRequest);
        setConflicts(foundConflicts);
      } catch (err) {
        console.error('Error checking conflicts:', err);
      }
    }
  };

  useEffect(() => {
    if (open && formData.date && formData.name) {
      const timeoutId = setTimeout(checkConflicts, 500);
      return () => clearTimeout(timeoutId);
    }
  }, [formData.date, formData.name, open]);

  const handleSave = async () => {
    setLoading(true);
    setError('');

    try {
      const holidayData: HolidayCreateRequest | HolidayUpdateRequest = {
        ...formData,
        date: formData.date.toISOString().split('T')[0],
      };

      // Validate the holiday
      const validation = await holidayService.validateHoliday(holidayData as HolidayCreateRequest);
      if (!validation.isValid) {
        setError(validation.errors.join(', '));
        setLoading(false);
        return;
      }

      let result: Holiday | null;
      if (holiday) {
        result = await holidayService.updateHoliday(holiday.id, { 
          id: holiday.id, 
          ...holidayData 
        } as HolidayUpdateRequest);
      } else {
        result = await holidayService.createHoliday(holidayData as HolidayCreateRequest);
      }

      if (result) {
        onSave(result);
        onClose();
      } else {
        setError('Failed to save holiday');
      }
    } catch (err: any) {
      setError(err.message || 'Failed to save holiday');
    } finally {
      setLoading(false);
    }
  };

  const isFormValid = () => {
    return formData.name.trim() && formData.date && formData.type;
  };

  const selectedDepartments = departments.filter(dept => 
    formData.departmentIds.includes(dept.id)
  );

  return (
    <LocalizationProvider dateAdapter={AdapterDateFns}>
      <Dialog
        open={open}
        onClose={onClose}
        maxWidth="md"
        fullWidth
        PaperProps={{
          sx: { minHeight: '60vh' }
        }}
      >
        <DialogTitle sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
          <Typography variant="h6">
            {holiday ? 'Edit Holiday' : 'Add Holiday'}
          </Typography>
          <IconButton onClick={onClose}>
            <Close />
          </IconButton>
        </DialogTitle>

        <DialogContent>
          {error && (
            <Alert severity="error" sx={{ mb: 2 }}>
              {error}
            </Alert>
          )}

          {conflicts.length > 0 && (
            <Alert severity="warning" sx={{ mb: 2 }}>
              <Typography variant="subtitle2" gutterBottom>
                Holiday conflicts detected:
              </Typography>
              {conflicts.map((conflict, index) => (
                <Box key={index} sx={{ mt: 1 }}>
                  <Typography variant="body2">
                    {conflict.date}: {conflict.holidays.map(h => h.name).join(', ')}
                  </Typography>
                </Box>
              ))}
            </Alert>
          )}

          <Box sx={{ display: 'flex', flexDirection: 'column', gap: 3 }}>
            {/* Basic Information */}
            <Box>
              <Typography variant="subtitle1" gutterBottom fontWeight="medium">
                Basic Information
              </Typography>
              
              <Box sx={{ display: 'flex', flexDirection: 'column', gap: 2 }}>
                <TextField
                  label="Holiday Name"
                  value={formData.name}
                  onChange={(e) => handleInputChange('name', e.target.value)}
                  fullWidth
                  required
                />

                <Box sx={{ display: 'flex', gap: 2, flexWrap: 'wrap' }}>
                  <Box sx={{ flex: '1 1 300px' }}>
                    <DatePicker
                      label="Date"
                      value={formData.date}
                      onChange={(newValue) => handleInputChange('date', newValue)}
                      slotProps={{
                        textField: {
                          fullWidth: true,
                          required: true,
                        },
                      }}
                    />
                  </Box>

                  <Box sx={{ flex: '1 1 200px' }}>
                    <FormControl fullWidth required>
                      <InputLabel>Type</InputLabel>
                      <Select
                        value={formData.type}
                        onChange={(e) => handleInputChange('type', e.target.value)}
                      >
                        {HOLIDAY_TYPES.map((type) => (
                          <MenuItem key={type.value} value={type.value}>
                            {type.label}
                          </MenuItem>
                        ))}
                      </Select>
                    </FormControl>
                  </Box>
                </Box>

                <TextField
                  label="Description"
                  value={formData.description}
                  onChange={(e) => handleInputChange('description', e.target.value)}
                  multiline
                  rows={3}
                  fullWidth
                />
              </Box>
            </Box>

            {/* Options */}
            <Box>
              <Typography variant="subtitle1" gutterBottom fontWeight="medium">
                Options
              </Typography>
              
              <Box sx={{ display: 'flex', flexDirection: 'column', gap: 2 }}>
                <FormControlLabel
                  control={
                    <Switch
                      checked={formData.isRecurring}
                      onChange={(e) => handleInputChange('isRecurring', e.target.checked)}
                    />
                  }
                  label="Recurring Holiday (yearly)"
                />

                {departments.length > 0 && (
                  <Autocomplete
                    multiple
                    options={departments}
                    getOptionLabel={(option) => option.name}
                    value={selectedDepartments}
                    onChange={(_, newValue) => 
                      handleInputChange('departmentIds', newValue.map(dept => dept.id))
                    }
                    renderTags={(value, getTagProps) =>
                      value.map((option, index) => (
                        <Chip
                          variant="outlined"
                          label={option.name}
                          {...getTagProps({ index })}
                          key={option.id}
                        />
                      ))
                    }
                    renderInput={(params) => (
                      <TextField
                        {...params}
                        label="Departments (optional)"
                        placeholder="Select departments"
                      />
                    )}
                  />
                )}
              </Box>
            </Box>

            {/* Color Selection */}
            <Box>
              <Typography variant="subtitle1" gutterBottom fontWeight="medium">
                <ColorLens sx={{ mr: 1, verticalAlign: 'middle' }} />
                Color
              </Typography>
              
              <Box sx={{ display: 'flex', gap: 1, flexWrap: 'wrap' }}>
                {HOLIDAY_COLORS.map((color) => (
                  <Box
                    key={color}
                    sx={{
                      width: 32,
                      height: 32,
                      backgroundColor: color,
                      borderRadius: 1,
                      cursor: 'pointer',
                      border: formData.color === color ? '3px solid' : '2px solid',
                      borderColor: formData.color === color ? 'primary.main' : 'divider',
                      '&:hover': {
                        borderColor: 'primary.main',
                        transform: 'scale(1.1)',
                      },
                      transition: 'all 0.2s',
                    }}
                    onClick={() => handleInputChange('color', color)}
                  />
                ))}
              </Box>
            </Box>
          </Box>
        </DialogContent>

        <DialogActions sx={{ p: 3 }}>
          <Button onClick={onClose}>
            Cancel
          </Button>
          <Button
            variant="contained"
            startIcon={loading ? <CircularProgress size={16} /> : <Save />}
            onClick={handleSave}
            disabled={!isFormValid() || loading}
          >
            {loading ? 'Saving...' : 'Save Holiday'}
          </Button>
        </DialogActions>
      </Dialog>
    </LocalizationProvider>
  );
};