import React, { useState, useEffect } from 'react';
import {
  Box,
  Typography,
  Button,
  TextField,
  Chip,
  Card,
  CardContent,
  Grid,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Avatar,
  IconButton,
  Tabs,
  Tab,
  Badge,
  Divider,
  SpeedDial,
  SpeedDialAction,
  SpeedDialIcon,
  useTheme,
  useMediaQuery,
} from '@mui/material';
import {
  Search,
  Add,
  Visibility,
  CheckCircle,
  Cancel,
  Pending,
  EventNote,
  CalendarMonth,
  TrendingUp,
  FilterList,
  GetApp,
  Close,
} from '@mui/icons-material';
import { LeaveRequest } from '../types';
import { leaveService } from '../services/leaveService';
import { format, differenceInDays } from 'date-fns';
import { MobileFilters } from '../components/Common/MobileFilters';
import { ResponsiveTable, MobileCard } from '../components/Common/MobileTable';

interface LeaveDialogProps {
  open: boolean;
  leave: LeaveRequest | null;
  onClose: () => void;
  onSave?: (leave: Partial<LeaveRequest>) => void;
  onApprove?: (leaveId: string, comments?: string) => void;
  onReject?: (leaveId: string, comments?: string) => void;
  mode: 'view' | 'edit' | 'create';
}

const LeaveDialog: React.FC<LeaveDialogProps> = ({
  open,
  leave,
  onClose,
  onSave,
  onApprove,
  onReject,
  mode,
}) => {
  const [formData, setFormData] = useState<Partial<LeaveRequest>>({
    employeeId: '',
    type: 'vacation',
    startDate: '',
    endDate: '',
    reason: '',
    status: 'pending',
  });
  const [comments, setComments] = useState('');
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down('md'));

  useEffect(() => {
    if (leave) {
      setFormData(leave);
      setComments(leave.comments || '');
    } else if (mode === 'create') {
      setFormData({
        employeeId: '',
        type: 'vacation',
        startDate: format(new Date(), 'yyyy-MM-dd'),
        endDate: format(new Date(), 'yyyy-MM-dd'),
        reason: '',
        status: 'pending',
      });
    }
  }, [leave, mode]);

  useEffect(() => {
    if (formData.startDate && formData.endDate) {
      const days = differenceInDays(new Date(formData.endDate), new Date(formData.startDate)) + 1;
      setFormData(prev => ({ ...prev, days: days > 0 ? days : 1 }));
    }
  }, [formData.startDate, formData.endDate]);

  const handleSubmit = () => {
    if (onSave) {
      onSave({
        ...formData,
        appliedDate: formData.appliedDate || format(new Date(), 'yyyy-MM-dd'),
      });
    }
  };

  const handleApprove = () => {
    if (onApprove && leave) {
      onApprove(leave.id, comments);
    }
  };

  const handleReject = () => {
    if (onReject && leave) {
      onReject(leave.id, comments);
    }
  };

  const isReadonly = mode === 'view' && leave?.status !== 'pending';

  return (
    <Dialog 
      open={open} 
      onClose={onClose} 
      maxWidth="md" 
      fullWidth
      fullScreen={isMobile}
      PaperProps={{
        sx: isMobile ? { 
          margin: 0,
          maxHeight: '100%',
          borderRadius: 0,
        } : undefined
      }}
    >
      <DialogTitle sx={{ 
        pb: 1,
        fontSize: { xs: '1.125rem', sm: '1.25rem' },
        display: 'flex',
        alignItems: 'center',
        justifyContent: 'space-between'
      }}>
        <span>
          {mode === 'create' ? 'Apply for Leave' : 
           mode === 'edit' ? 'Edit Leave Request' : 'Leave Request Details'}
        </span>
        {isMobile && (
          <IconButton onClick={onClose} size="small">
            <Close />
          </IconButton>
        )}
      </DialogTitle>
      <DialogContent sx={{ px: { xs: 2, sm: 3 } }}>
        <Grid container spacing={2} sx={{ mt: 1 }}>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Employee ID"
              value={formData.employeeId || ''}
              onChange={(e) => setFormData({ ...formData, employeeId: e.target.value })}
              disabled={mode === 'view'}
              required
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <FormControl fullWidth size={isMobile ? "medium" : "small"}>
              <InputLabel>Leave Type</InputLabel>
              <Select
                value={formData.type || 'vacation'}
                onChange={(e) => setFormData({ ...formData, type: e.target.value as any })}
                disabled={mode === 'view'}
                label="Leave Type"
              >
                <MenuItem value="sick">Sick Leave</MenuItem>
                <MenuItem value="vacation">Vacation</MenuItem>
                <MenuItem value="personal">Personal Leave</MenuItem>
                <MenuItem value="maternity">Maternity Leave</MenuItem>
                <MenuItem value="paternity">Paternity Leave</MenuItem>
              </Select>
            </FormControl>
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Start Date"
              type="date"
              value={formData.startDate || ''}
              onChange={(e) => setFormData({ ...formData, startDate: e.target.value })}
              disabled={mode === 'view'}
              InputLabelProps={{ shrink: true }}
              required
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="End Date"
              type="date"
              value={formData.endDate || ''}
              onChange={(e) => setFormData({ ...formData, endDate: e.target.value })}
              disabled={mode === 'view'}
              InputLabelProps={{ shrink: true }}
              required
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Total Days"
              value={formData.days || ''}
              disabled
              InputProps={{
                readOnly: true,
              }}
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Applied Date"
              type="date"
              value={formData.appliedDate || ''}
              disabled
              InputLabelProps={{ shrink: true }}
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          <Grid size={12}>
            <TextField
              fullWidth
              label="Reason"
              multiline
              rows={3}
              value={formData.reason || ''}
              onChange={(e) => setFormData({ ...formData, reason: e.target.value })}
              disabled={mode === 'view'}
              required
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          {leave && leave.status !== 'pending' && (
            <Grid size={12}>
              <TextField
                fullWidth
                label="Status"
                value={formData.status}
                disabled
                size={isMobile ? "medium" : "small"}
              />
            </Grid>
          )}
          {(mode === 'view' && leave?.status === 'pending') && (
            <Grid size={12}>
              <Divider sx={{ my: 2 }} />
              <Typography variant="h6" gutterBottom>
                Management Action
              </Typography>
              <TextField
                fullWidth
                label="Comments"
                multiline
                rows={2}
                value={comments}
                onChange={(e) => setComments(e.target.value)}
                placeholder="Add comments for approval/rejection"
                size={isMobile ? "medium" : "small"}
              />
            </Grid>
          )}
          {leave?.comments && (
            <Grid size={12}>
              <TextField
                fullWidth
                label="Management Comments"
                multiline
                rows={2}
                value={leave.comments}
                disabled
                size={isMobile ? "medium" : "small"}
              />
            </Grid>
          )}
        </Grid>
      </DialogContent>
      <DialogActions sx={{ 
        p: { xs: 2, sm: 3 }, 
        gap: 1,
        flexDirection: { xs: 'column', sm: 'row' }
      }}>
        <Button onClick={onClose} fullWidth={isMobile}>
          Close
        </Button>
        {mode === 'view' && leave?.status === 'pending' && (
          <>
            <Button 
              onClick={handleReject} 
              color="error" 
              fullWidth={isMobile}
            >
              Reject
            </Button>
            <Button 
              onClick={handleApprove} 
              variant="contained" 
              color="success"
              fullWidth={isMobile}
            >
              Approve
            </Button>
          </>
        )}
        {mode !== 'view' && (
          <Button 
            onClick={handleSubmit} 
            variant="contained"
            fullWidth={isMobile}
          >
            {mode === 'create' ? 'Submit Request' : 'Save Changes'}
          </Button>
        )}
      </DialogActions>
    </Dialog>
  );
};

interface TabPanelProps {
  children?: React.ReactNode;
  index: number;
  value: number;
}

const TabPanel: React.FC<TabPanelProps> = ({ children, value, index }) => (
  <div hidden={value !== index}>
    {value === index && <Box sx={{ py: { xs: 2, sm: 3 } }}>{children}</Box>}
  </div>
);

export const MobileLeaves: React.FC = () => {
  const [leaves, setLeaves] = useState<LeaveRequest[]>([]);
  const [filteredLeaves, setFilteredLeaves] = useState<LeaveRequest[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState('');
  const [typeFilter, setTypeFilter] = useState('');
  const [selectedLeave, setSelectedLeave] = useState<LeaveRequest | null>(null);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [dialogMode, setDialogMode] = useState<'view' | 'edit' | 'create'>('view');
  const [tabValue, setTabValue] = useState(0);
  
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down('md'));

  const mockEmployeeNames: { [key: string]: string } = {
    '1': 'John Doe',
    '2': 'Sarah Johnson',
    '3': 'Michael Brown',
    '4': 'Emily Davis',
    '5': 'David Wilson',
  };

  useEffect(() => {
    loadLeaves();
  }, []);

  useEffect(() => {
    filterLeaves();
  }, [leaves, searchTerm, statusFilter, typeFilter, tabValue]);

  const loadLeaves = async () => {
    try {
      setLoading(true);
      const data = await leaveService.getAllLeaves();
      setLeaves(data);
    } catch (error) {
      console.error('Failed to load leaves:', error);
    } finally {
      setLoading(false);
    }
  };

  const filterLeaves = () => {
    let filtered = leaves;

    // Filter by tab
    if (tabValue === 1) {
      filtered = filtered.filter(leave => leave.status === 'pending');
    }

    if (searchTerm) {
      filtered = filtered.filter(leave =>
        mockEmployeeNames[leave.employeeId]?.toLowerCase().includes(searchTerm.toLowerCase()) ||
        leave.employeeId.toLowerCase().includes(searchTerm.toLowerCase()) ||
        leave.reason.toLowerCase().includes(searchTerm.toLowerCase())
      );
    }

    if (statusFilter) {
      filtered = filtered.filter(leave => leave.status === statusFilter);
    }

    if (typeFilter) {
      filtered = filtered.filter(leave => leave.type === typeFilter);
    }

    setFilteredLeaves(filtered);
  };

  const handleView = (leave: LeaveRequest) => {
    setSelectedLeave(leave);
    setDialogMode('view');
    setDialogOpen(true);
  };

  const handleCreate = () => {
    setSelectedLeave(null);
    setDialogMode('create');
    setDialogOpen(true);
  };

  const handleSave = async (leaveData: Partial<LeaveRequest>) => {
    try {
      if (dialogMode === 'create') {
        await leaveService.createLeaveRequest(leaveData as Omit<LeaveRequest, 'id' | 'appliedDate' | 'status'>);
      }
      setDialogOpen(false);
      loadLeaves();
    } catch (error) {
      console.error('Failed to save leave:', error);
    }
  };

  const handleApprove = async (leaveId: string, comments?: string) => {
    try {
      await leaveService.approveLeave(leaveId, 'MANAGER001', comments);
      setDialogOpen(false);
      loadLeaves();
    } catch (error) {
      console.error('Failed to approve leave:', error);
    }
  };

  const handleReject = async (leaveId: string, comments?: string) => {
    try {
      await leaveService.rejectLeave(leaveId, 'MANAGER001', comments);
      setDialogOpen(false);
      loadLeaves();
    } catch (error) {
      console.error('Failed to reject leave:', error);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'approved': return 'success';
      case 'rejected': return 'error';
      case 'pending': return 'warning';
      default: return 'default';
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'approved': return <CheckCircle />;
      case 'rejected': return <Cancel />;
      case 'pending': return <Pending />;
      default: return <Pending />;
    }
  };

  const getLeaveTypeColor = (type: string) => {
    switch (type) {
      case 'sick': return 'error';
      case 'vacation': return 'primary';
      case 'personal': return 'info';
      case 'maternity': return 'secondary';
      case 'paternity': return 'secondary';
      default: return 'default';
    }
  };

  const leaveStats = {
    total: leaves.length,
    pending: leaves.filter(l => l.status === 'pending').length,
    approved: leaves.filter(l => l.status === 'approved').length,
    rejected: leaves.filter(l => l.status === 'rejected').length,
    totalDays: leaves.filter(l => l.status === 'approved').reduce((sum, l) => sum + l.days, 0),
  };

  const filters = [
    {
      key: 'status',
      label: 'Status',
      type: 'select' as const,
      value: statusFilter,
      options: [
        { value: 'pending', label: 'Pending' },
        { value: 'approved', label: 'Approved' },
        { value: 'rejected', label: 'Rejected' }
      ]
    },
    {
      key: 'type',
      label: 'Type',
      type: 'select' as const,
      value: typeFilter,
      options: [
        { value: 'sick', label: 'Sick' },
        { value: 'vacation', label: 'Vacation' },
        { value: 'personal', label: 'Personal' },
        { value: 'maternity', label: 'Maternity' },
        { value: 'paternity', label: 'Paternity' }
      ]
    }
  ];

  const handleFilterChange = (key: string, value: string) => {
    if (key === 'status') setStatusFilter(value);
    if (key === 'type') setTypeFilter(value);
  };

  const handleClearFilters = () => {
    setSearchTerm('');
    setStatusFilter('');
    setTypeFilter('');
  };

  const columns = [
    {
      field: 'employee',
      label: 'Employee',
      render: (value: any, row: LeaveRequest) => (
        <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
          <Avatar sx={{ width: 32, height: 32 }}>
            {mockEmployeeNames[row.employeeId]?.split(' ').map(n => n[0]).join('') || row.employeeId[0]}
          </Avatar>
          <Box>
            <Typography variant="body2" fontWeight="bold">
              {mockEmployeeNames[row.employeeId] || `Employee ${row.employeeId}`}
            </Typography>
            <Typography variant="body2" color="textSecondary">
              ID: {row.employeeId}
            </Typography>
          </Box>
        </Box>
      ),
    },
    {
      field: 'type',
      label: 'Type',
      render: (value: string) => (
        <Chip
          label={value}
          color={getLeaveTypeColor(value) as any}
          size="small"
          variant="outlined"
        />
      ),
    },
    { 
      field: 'duration', 
      label: 'Duration',
      render: (value: any, row: LeaveRequest) => (
        <Typography variant="body2">
          {format(new Date(row.startDate), 'MMM dd')} - {format(new Date(row.endDate), 'MMM dd, yyyy')}
        </Typography>
      )
    },
    { 
      field: 'days', 
      label: 'Days',
      render: (value: number) => (
        <Typography variant="body2" fontWeight="bold">
          {value} days
        </Typography>
      )
    },
    { 
      field: 'appliedDate', 
      label: 'Applied',
      render: (value: string) => format(new Date(value), 'MMM dd, yyyy')
    },
    {
      field: 'status',
      label: 'Status',
      render: (value: string) => (
        <Chip
          icon={getStatusIcon(value)}
          label={value}
          color={getStatusColor(value) as any}
          size="small"
        />
      ),
    },
    {
      field: 'actions',
      label: 'Actions',
      render: (value: any, row: LeaveRequest) => (
        <IconButton onClick={() => handleView(row)} size="small">
          <Visibility />
        </IconButton>
      ),
    },
  ];

  const mobileCardConfig = {
    title: (row: LeaveRequest) => mockEmployeeNames[row.employeeId] || `Employee ${row.employeeId}`,
    subtitle: (row: LeaveRequest) => `${format(new Date(row.startDate), 'MMM dd')} - ${format(new Date(row.endDate), 'MMM dd')} • ${row.days} days`,
    avatar: (row: LeaveRequest) => mockEmployeeNames[row.employeeId]?.split(' ').map(n => n[0]).join('') || row.employeeId[0],
    status: (row: LeaveRequest) => ({
      label: row.status,
      color: getStatusColor(row.status) as any,
    }),
    mainFields: ['type', 'appliedDate'],
    expandableFields: ['reason'],
    actions: (row: LeaveRequest) => [
      <IconButton key="view" onClick={() => handleView(row)} size="small">
        <Visibility fontSize="small" />
      </IconButton>,
    ],
  };

  const speedDialActions = [
    {
      icon: <Add />,
      name: 'Apply Leave',
      onClick: handleCreate,
    },
    {
      icon: <GetApp />,
      name: 'Export',
      onClick: () => console.log('Export'),
    },
    {
      icon: <CalendarMonth />,
      name: 'Calendar View',
      onClick: () => console.log('Calendar View'),
    },
  ];

  const statsCards = [
    { title: 'Total', value: leaveStats.total, icon: <EventNote />, color: '#1976d2' },
    { title: 'Pending', value: leaveStats.pending, icon: <Pending />, color: '#ed6c02' },
    { title: 'Approved', value: leaveStats.approved, icon: <CheckCircle />, color: '#2e7d32' },
    { title: 'Days Used', value: leaveStats.totalDays, icon: <CalendarMonth />, color: '#1976d2' },
  ];

  return (
    <Box>
      <Box sx={{ 
        display: 'flex', 
        justifyContent: 'space-between', 
        alignItems: 'center', 
        mb: { xs: 2, sm: 3 },
        flexDirection: { xs: 'column', sm: 'row' },
        gap: { xs: 2, sm: 0 }
      }}>
        <Typography 
          variant={isMobile ? "h5" : "h4"} 
          sx={{ fontWeight: 600 }}
        >
          Leave Management
        </Typography>
        {!isMobile && (
          <Button
            variant="contained"
            startIcon={<Add />}
            onClick={handleCreate}
            sx={{ minWidth: 140 }}
          >
            Apply Leave
          </Button>
        )}
      </Box>
      <Box sx={{ borderBottom: 1, borderColor: 'divider', mb: 2 }}>
        <Tabs 
          value={tabValue} 
          onChange={(e, newValue) => setTabValue(newValue)}
          variant={isMobile ? "scrollable" : "standard"}
          scrollButtons={isMobile ? "auto" : false}
          allowScrollButtonsMobile
        >
          <Tab label="All Requests" />
          <Tab 
            label={
              isMobile ? (
                <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                  Pending
                  {leaveStats.pending > 0 && (
                    <Chip 
                      label={leaveStats.pending} 
                      size="small" 
                      color="warning" 
                      sx={{ height: 16, fontSize: '0.6rem' }} 
                    />
                  )}
                </Box>
              ) : (
                <Badge badgeContent={leaveStats.pending} color="warning">
                  Pending Approval
                </Badge>
              )
            } 
          />
          <Tab label="Analytics" />
        </Tabs>
      </Box>
      <TabPanel value={tabValue} index={0}>
        {/* Stats Cards */}
        <Grid container spacing={2} sx={{ mb: 3 }}>
          {statsCards.map((stat, index) => (
            <Grid
              key={index}
              size={{
                xs: 6,
                sm: 3
              }}>
              <Card>
                <CardContent sx={{ 
                  textAlign: 'center',
                  py: { xs: 1.5, sm: 2 },
                  px: { xs: 1, sm: 2 }
                }}>
                  <Box sx={{ 
                    display: 'flex', 
                    flexDirection: 'column', 
                    alignItems: 'center',
                    gap: { xs: 0.5, sm: 1 }
                  }}>
                    {React.cloneElement(stat.icon, { 
                      sx: { 
                        fontSize: { xs: 24, sm: 40 },
                        color: stat.color,
                        mb: { xs: 0.5, sm: 1 }
                      }
                    })}
                    <Typography 
                      variant="h5" 
                      sx={{ 
                        fontSize: { xs: '1.25rem', sm: '2rem' },
                        fontWeight: 600 
                      }}
                    >
                      {stat.value}
                    </Typography>
                    <Typography 
                      variant="body2" 
                      color="textSecondary"
                      sx={{ fontSize: { xs: '0.75rem', sm: '0.875rem' } }}
                    >
                      {stat.title}
                    </Typography>
                  </Box>
                </CardContent>
              </Card>
            </Grid>
          ))}
        </Grid>

        {/* Filters */}
        <MobileFilters
          searchValue={searchTerm}
          onSearchChange={setSearchTerm}
          filters={filters}
          onFilterChange={handleFilterChange}
          onClearFilters={handleClearFilters}
          placeholder="Search requests..."
        />

        {/* Leave Requests List/Table */}
        <ResponsiveTable
          columns={columns}
          data={filteredLeaves}
          mobileCardConfig={mobileCardConfig}
          onRowClick={isMobile ? handleView : undefined}
        />
      </TabPanel>
      <TabPanel value={tabValue} index={1}>
        {/* Pending requests only */}
        <Typography variant="h6" gutterBottom>
          Requests Awaiting Approval ({leaveStats.pending})
        </Typography>
        
        <ResponsiveTable
          columns={columns}
          data={filteredLeaves}
          mobileCardConfig={mobileCardConfig}
          onRowClick={isMobile ? handleView : undefined}
        />
      </TabPanel>
      <TabPanel value={tabValue} index={2}>
        <Typography variant="h6" gutterBottom>
          Leave Analytics
        </Typography>
        <Typography variant="body1" color="textSecondary">
          Analytics and reporting features will be implemented here.
        </Typography>
      </TabPanel>
      {/* Mobile Speed Dial */}
      {isMobile && (
        <SpeedDial
          ariaLabel="Leave actions"
          sx={{ position: 'fixed', bottom: 24, right: 24 }}
          icon={<SpeedDialIcon />}
        >
          {speedDialActions.map((action) => (
            <SpeedDialAction
              key={action.name}
              icon={action.icon}
              tooltipTitle={action.name}
              onClick={action.onClick}
            />
          ))}
        </SpeedDial>
      )}
      <LeaveDialog
        open={dialogOpen}
        leave={selectedLeave}
        onClose={() => setDialogOpen(false)}
        onSave={handleSave}
        onApprove={handleApprove}
        onReject={handleReject}
        mode={dialogMode}
      />
    </Box>
  );
};