import { DashboardStats } from '../index';

// Dashboard service specific types
export interface DashboardFilter {
  dateRange?: {
    startDate: string;
    endDate: string;
  };
  departments?: string[];
  includeInactive?: boolean;
}

export interface DashboardWidget {
  id: string;
  type: 'stat' | 'chart' | 'table' | 'progress' | 'activity' | 'calendar';
  title: string;
  description?: string;
  size: 'small' | 'medium' | 'large' | 'full';
  position: {
    x: number;
    y: number;
    width: number;
    height: number;
  };
  dataSource: string;
  config: Record<string, any>;
  refreshInterval?: number; // seconds
  isVisible: boolean;
  permissions?: string[];
}

export interface DashboardLayout {
  id: string;
  name: string;
  description?: string;
  widgets: DashboardWidget[];
  isDefault: boolean;
  userRole?: string;
  createdBy: string;
  createdAt: string;
  isShared: boolean;
}

export interface UserDashboardConfig {
  userId: string;
  layoutId: string;
  customWidgets: DashboardWidget[];
  preferences: {
    autoRefresh: boolean;
    refreshInterval: number;
    theme: 'light' | 'dark' | 'auto';
    compactMode: boolean;
  };
  lastModified: string;
}

// Enhanced Dashboard Stats
export interface ExtendedDashboardStats extends DashboardStats {
  previousPeriodStats?: DashboardStats;
  trends: {
    employeeGrowthRate: number;
    attendanceTrend: number;
    leavesTrend: number;
    payrollTrend: number;
  };
  alerts: DashboardAlert[];
}

export interface DashboardAlert {
  id: string;
  type: 'info' | 'warning' | 'error' | 'success';
  priority: 'low' | 'medium' | 'high' | 'urgent';
  title: string;
  message: string;
  category: 'attendance' | 'leave' | 'payroll' | 'employee' | 'system';
  actionRequired: boolean;
  actionUrl?: string;
  createdAt: string;
  expiresAt?: string;
}

// Metrics and KPIs
export interface DashboardMetrics {
  overview: {
    employeeMetrics: EmployeeMetrics;
    attendanceMetrics: AttendanceMetrics;
    leaveMetrics: LeaveMetrics;
    payrollMetrics: PayrollMetrics;
    performanceMetrics: PerformanceMetrics;
  };
  trends: {
    employeeTrends: TimeSeries[];
    attendanceTrends: TimeSeries[];
    leaveTrends: TimeSeries[];
    payrollTrends: TimeSeries[];
  };
  comparisons: {
    departmentComparison: DepartmentComparison[];
    periodComparison: PeriodComparison;
  };
  forecasts: {
    employeeGrowth: Forecast[];
    payrollProjection: Forecast[];
    leaveUtilization: Forecast[];
  };
}

export interface EmployeeMetrics {
  total: number;
  active: number;
  inactive: number;
  newHires: number;
  terminations: number;
  turnoverRate: number;
  avgTenure: number; // months
  diversityMetrics: {
    genderRatio: Array<{ gender: string; count: number; percentage: number }>;
    ageDistribution: Array<{ ageGroup: string; count: number }>;
    departmentDistribution: Array<{ department: string; count: number }>;
  };
}

export interface AttendanceMetrics {
  overallRate: number;
  punctualityRate: number;
  avgWorkingHours: number;
  totalOvertimeHours: number;
  absenteeismRate: number;
  patternAnalysis: {
    peakAttendanceDays: string[];
    commonAbsentDays: string[];
    seasonalTrends: Array<{
      season: string;
      avgRate: number;
    }>;
  };
}

export interface LeaveMetrics {
  totalRequests: number;
  pendingRequests: number;
  approvalRate: number;
  avgProcessingTime: number; // days
  utilizationRate: number;
  typeBreakdown: Array<{
    type: string;
    count: number;
    avgDuration: number;
  }>;
  seasonalPatterns: Array<{
    month: string;
    requestCount: number;
  }>;
}

export interface PayrollMetrics {
  totalCost: number;
  avgSalary: number;
  costPerEmployee: number;
  overtimeCost: number;
  benefitsCost: number;
  taxBurden: number;
  departmentCosts: Array<{
    department: string;
    totalCost: number;
    employeeCount: number;
    avgCost: number;
  }>;
}

export interface PerformanceMetrics {
  avgRating: number;
  reviewCompletion: number;
  goalCompletion: number;
  improvementPlans: number;
  topPerformers: number;
  performanceDistribution: Array<{
    rating: string;
    count: number;
    percentage: number;
  }>;
}

export interface TimeSeries {
  date: string;
  value: number;
  change?: number;
  trend?: 'up' | 'down' | 'stable';
}

export interface DepartmentComparison {
  department: string;
  metrics: {
    employeeCount: number;
    attendanceRate: number;
    leaveUtilization: number;
    avgSalary: number;
    performanceRating: number;
    turnoverRate: number;
  };
  ranking: number;
  improvements: string[];
}

export interface PeriodComparison {
  currentPeriod: {
    startDate: string;
    endDate: string;
    stats: DashboardStats;
  };
  previousPeriod: {
    startDate: string;
    endDate: string;
    stats: DashboardStats;
  };
  changes: {
    employeeChange: number;
    attendanceChange: number;
    leaveChange: number;
    payrollChange: number;
  };
}

export interface Forecast {
  date: string;
  predicted: number;
  confidence: number; // 0-1
  lowerBound: number;
  upperBound: number;
}

// Activity and Recent Changes
export interface DashboardActivity {
  id: string;
  type: 'employee_joined' | 'employee_left' | 'leave_approved' | 'payroll_processed' | 'attendance_alert';
  title: string;
  description: string;
  timestamp: string;
  userId?: string;
  userName?: string;
  metadata?: Record<string, any>;
  priority: 'low' | 'medium' | 'high';
}

export interface RecentChanges {
  newEmployees: Array<{
    employeeId: string;
    name: string;
    department: string;
    joinDate: string;
  }>;
  recentTerminations: Array<{
    employeeId: string;
    name: string;
    department: string;
    terminationDate: string;
  }>;
  pendingApprovals: Array<{
    type: 'leave' | 'expense' | 'timeoff';
    count: number;
    oldestRequest: string;
  }>;
  upcomingEvents: Array<{
    type: 'birthday' | 'anniversary' | 'review' | 'holiday';
    date: string;
    description: string;
  }>;
}

// Chart Data Types
export interface ChartData {
  labels: string[];
  datasets: Array<{
    label: string;
    data: number[];
    backgroundColor?: string | string[];
    borderColor?: string;
    borderWidth?: number;
  }>;
}

export interface BarChartData extends ChartData {}
export interface LineChartData extends ChartData {}
export interface PieChartData extends ChartData {}
export interface AreaChartData extends ChartData {}

export interface DonutChartData {
  labels: string[];
  data: number[];
  colors: string[];
  total?: number;
}

// Quick Actions
export interface QuickAction {
  id: string;
  title: string;
  description: string;
  icon: string;
  url: string;
  category: 'employee' | 'attendance' | 'leave' | 'payroll' | 'reports';
  permissions?: string[];
  isVisible: boolean;
  order: number;
}

// Dashboard Export
export interface DashboardExportOptions {
  format: 'pdf' | 'excel' | 'csv';
  includeCharts: boolean;
  includeRawData: boolean;
  dateRange?: {
    startDate: string;
    endDate: string;
  };
  widgets?: string[]; // Widget IDs to include
}

export interface DashboardExportResult {
  success: boolean;
  downloadUrl?: string;
  filename: string;
  format: string;
  error?: string;
}

// Real-time Updates
export interface DashboardUpdate {
  type: 'metric_update' | 'new_activity' | 'alert' | 'widget_refresh';
  widgetId?: string;
  data: any;
  timestamp: string;
}

export interface DashboardSubscription {
  userId: string;
  widgets: string[];
  updateFrequency: 'realtime' | '1min' | '5min' | '15min' | '1hour';
  isActive: boolean;
}

// Custom Dashboard Queries
export interface CustomQuery {
  id: string;
  name: string;
  description: string;
  sql: string;
  parameters: Array<{
    name: string;
    type: 'string' | 'number' | 'date' | 'boolean';
    defaultValue?: any;
  }>;
  resultColumns: Array<{
    name: string;
    type: string;
    displayName: string;
  }>;
  createdBy: string;
  isShared: boolean;
  lastExecuted?: string;
}

export interface QueryResult {
  columns: string[];
  rows: any[][];
  executionTime: number;
  recordCount: number;
  error?: string;
}