// Notification service specific types
export interface NotificationFilter {
  userId?: string;
  type?: NotificationBase['type'];
  category?: NotificationBase['category'];
  isRead?: boolean;
  priority?: NotificationBase['priority'];
  dateRange?: {
    startDate: string;
    endDate: string;
  };
  channels?: NotificationChannel[];
}

export interface NotificationSortOptions {
  field: 'createdAt' | 'priority' | 'type';
  order: 'asc' | 'desc';
}

export interface NotificationPaginationOptions {
  page: number;
  limit: number;
  sort?: NotificationSortOptions;
}

export interface NotificationListResponse {
  notifications: NotificationBase[];
  total: number;
  unreadCount: number;
  page: number;
  limit: number;
  totalPages: number;
}

export interface NotificationBase {
  id: string;
  type: 'info' | 'success' | 'warning' | 'error';
  category: 'system' | 'attendance' | 'leave' | 'payroll' | 'performance' | 'announcement';
  priority: 'low' | 'medium' | 'high' | 'urgent';
  title: string;
  message: string;
  userId?: string; // null for broadcast notifications
  createdAt: string;
  readAt?: string;
  isRead: boolean;
  metadata?: {
    resourceId?: string;
    resourceType?: string;
    actionUrl?: string;
    actionLabel?: string;
    expiresAt?: string;
    tags?: string[];
  };
  channels: NotificationChannel[];
  deliveryStatus: NotificationDeliveryStatus[];
}

export type NotificationChannel = 'in_app' | 'email' | 'sms' | 'push' | 'slack';

export interface NotificationDeliveryStatus {
  channel: NotificationChannel;
  status: 'pending' | 'sent' | 'delivered' | 'failed' | 'bounced';
  attemptCount: number;
  lastAttemptAt?: string;
  deliveredAt?: string;
  error?: string;
}

export interface NotificationCreateRequest {
  type: NotificationBase['type'];
  category: NotificationBase['category'];
  priority: NotificationBase['priority'];
  title: string;
  message: string;
  userId?: string;
  userIds?: string[]; // For multiple recipients
  channels: NotificationChannel[];
  metadata?: NotificationBase['metadata'];
  scheduleFor?: string; // ISO date string for scheduled notifications
  template?: {
    templateId: string;
    variables: Record<string, any>;
  };
}

export interface NotificationBulkCreateRequest {
  notifications: NotificationCreateRequest[];
  batchSize?: number;
}

export interface NotificationBulkOperationResult {
  success: boolean;
  createdCount: number;
  failedCount: number;
  errors?: Array<{
    index: number;
    error: string;
  }>;
}

export interface NotificationTemplate {
  id: string;
  name: string;
  description: string;
  category: NotificationBase['category'];
  subject: string;
  bodyTemplate: string; // Supports template variables like {{variableName}}
  channels: NotificationChannel[];
  variables: Array<{
    name: string;
    description: string;
    type: 'string' | 'number' | 'date' | 'boolean';
    required: boolean;
    defaultValue?: any;
  }>;
  isActive: boolean;
  createdBy: string;
  createdAt: string;
  usageCount: number;
}

export interface NotificationPreferences {
  userId: string;
  channels: {
    [K in NotificationChannel]: boolean;
  };
  categoryPreferences: {
    [K in NotificationBase['category']]: {
      enabled: boolean;
      channels: NotificationChannel[];
      priority: NotificationBase['priority'];
    };
  };
  quietHours: {
    enabled: boolean;
    start: string; // HH:MM
    end: string; // HH:MM
    timezone: string;
  };
  frequency: 'immediate' | 'hourly' | 'daily' | 'weekly';
  digestSettings: {
    enabled: boolean;
    frequency: 'daily' | 'weekly';
    time: string; // HH:MM
    maxItems: number;
  };
}

export interface NotificationRule {
  id: string;
  name: string;
  description: string;
  isActive: boolean;
  trigger: {
    event: string; // e.g., 'employee.created', 'leave.approved'
    conditions: Array<{
      field: string;
      operator: 'equals' | 'not_equals' | 'greater_than' | 'less_than' | 'contains';
      value: any;
    }>;
  };
  action: {
    templateId: string;
    channels: NotificationChannel[];
    recipients: {
      type: 'specific' | 'role' | 'department' | 'all';
      userIds?: string[];
      roles?: string[];
      departments?: string[];
    };
    delay?: number; // Minutes to delay notification
  };
  createdBy: string;
  createdAt: string;
}

export interface NotificationTriggerEvent {
  id: string;
  event: string;
  data: Record<string, any>;
  userId?: string;
  triggeredBy: string;
  triggeredAt: string;
  processedAt?: string;
  status: 'pending' | 'processing' | 'completed' | 'failed';
  notificationsCreated: number;
  error?: string;
}

export interface NotificationStats {
  total: number;
  unread: number;
  byType: {
    [K in NotificationBase['type']]: number;
  };
  byCategory: {
    [K in NotificationBase['category']]: number;
  };
  byPriority: {
    [K in NotificationBase['priority']]: number;
  };
  deliveryStats: {
    [K in NotificationChannel]: {
      sent: number;
      delivered: number;
      failed: number;
      deliveryRate: number;
    };
  };
  recentActivity: Array<{
    date: string;
    sent: number;
    delivered: number;
    read: number;
  }>;
}

export interface NotificationDigest {
  id: string;
  userId: string;
  period: {
    startDate: string;
    endDate: string;
  };
  notifications: Array<{
    id: string;
    title: string;
    message: string;
    type: NotificationBase['type'];
    category: NotificationBase['category'];
    createdAt: string;
  }>;
  summary: {
    totalCount: number;
    highPriorityCount: number;
    categoryCounts: {
      [K in NotificationBase['category']]: number;
    };
  };
  generatedAt: string;
  sentAt?: string;
  deliveryStatus: NotificationDeliveryStatus[];
}

export interface NotificationSubscription {
  id: string;
  userId: string;
  eventType: string;
  isActive: boolean;
  filters: Record<string, any>;
  channels: NotificationChannel[];
  createdAt: string;
}

export interface NotificationWebhook {
  id: string;
  name: string;
  url: string;
  events: string[];
  headers?: Record<string, string>;
  isActive: boolean;
  secret?: string;
  retryPolicy: {
    maxRetries: number;
    retryInterval: number; // seconds
    backoffMultiplier: number;
  };
  lastTriggered?: string;
  successCount: number;
  failureCount: number;
}

export interface NotificationQueue {
  id: string;
  notification: NotificationBase;
  scheduledFor: string;
  attempts: number;
  maxAttempts: number;
  status: 'queued' | 'processing' | 'completed' | 'failed' | 'cancelled';
  error?: string;
  createdAt: string;
  processedAt?: string;
}

export interface NotificationAuditLog {
  id: string;
  notificationId: string;
  action: 'created' | 'sent' | 'delivered' | 'read' | 'deleted' | 'failed';
  channel?: NotificationChannel;
  userId?: string;
  details?: Record<string, any>;
  timestamp: string;
  ipAddress?: string;
  userAgent?: string;
}

export interface NotificationCampaign {
  id: string;
  name: string;
  description: string;
  templateId: string;
  audience: {
    type: 'all' | 'role' | 'department' | 'custom';
    roles?: string[];
    departments?: string[];
    userIds?: string[];
  };
  scheduling: {
    type: 'immediate' | 'scheduled' | 'recurring';
    scheduleFor?: string;
    recurringPattern?: {
      frequency: 'daily' | 'weekly' | 'monthly';
      interval: number;
      endDate?: string;
    };
  };
  status: 'draft' | 'scheduled' | 'running' | 'completed' | 'cancelled';
  metrics: {
    totalRecipients: number;
    sent: number;
    delivered: number;
    opened: number;
    clicked: number;
    failed: number;
  };
  createdBy: string;
  createdAt: string;
  sentAt?: string;
}

export interface NotificationAnalytics {
  period: {
    startDate: string;
    endDate: string;
  };
  overview: NotificationStats;
  trends: Array<{
    date: string;
    sent: number;
    delivered: number;
    opened: number;
    failed: number;
  }>;
  topCategories: Array<{
    category: NotificationBase['category'];
    count: number;
    openRate: number;
  }>;
  channelPerformance: Array<{
    channel: NotificationChannel;
    sent: number;
    deliveryRate: number;
    openRate: number;
    avgDeliveryTime: number; // minutes
  }>;
  userEngagement: Array<{
    userId: string;
    userName: string;
    totalReceived: number;
    totalRead: number;
    readRate: number;
    lastActivity: string;
  }>;
}