# HR Management System - TypeScript Types Documentation

This document provides an overview of all TypeScript types created for the HR Management System, organized by service.

## Type Organization Structure

```
src/types/
├── index.ts                      # Main types export file
├── serviceInterfaces.ts          # Service interface contracts
├── api.types.ts                 # API request/response types
└── services/
    ├── index.ts                 # Service types export file
    ├── employeeService.types.ts # Employee service types
    ├── departmentService.types.ts # Department service types
    ├── attendanceService.types.ts # Attendance service types
    ├── leaveService.types.ts    # Leave service types
    ├── payrollService.types.ts  # Payroll service types
    ├── userService.types.ts     # User service types
    ├── reportsService.types.ts  # Reports service types
    ├── notificationService.types.ts # Notification service types
    └── dashboardService.types.ts # Dashboard service types
```

## Core Entity Types

### Base Entities
- `User` - User account information
- `Employee` - Employee data with personal and professional details
- `Department` - Department information with budget and management details
- `Attendance` - Daily attendance records with check-in/out times
- `LeaveRequest` - Leave application and approval workflow
- `PayrollRecord` - Salary calculation and payment records
- `DashboardStats` - System overview metrics

## Service-Specific Types

### 1. Employee Service Types (`employeeService.types.ts`)

**Core Types:**
- `EmployeeFilter` - Filtering options for employee queries
- `EmployeeSortOptions` - Sorting configurations
- `EmployeePaginationOptions` - Pagination parameters
- `EmployeeListResponse` - Paginated employee list response
- `EmployeeCreateRequest` - Employee creation payload
- `EmployeeUpdateRequest` - Employee update payload
- `EmployeeStats` - Employee statistics and metrics

**Advanced Types:**
- `EmployeeBulkOperation` - Bulk operations on multiple employees
- `EmployeeBulkOperationResult` - Results of bulk operations
- `EmployeeSearchResult` - Search functionality results
- `EmployeeValidationResult` - Validation feedback
- `EmployeePerformanceMetrics` - Performance tracking data

### 2. Department Service Types (`departmentService.types.ts`)

**Core Types:**
- `DepartmentFilter` - Department filtering options
- `DepartmentStats` - Department-level statistics
- `DepartmentBudgetAnalysis` - Budget utilization tracking
- `DepartmentEmployeeDistribution` - Employee distribution analysis

**Advanced Types:**
- `DepartmentPerformanceMetrics` - Performance indicators
- `DepartmentReorganization` - Organizational changes
- `DepartmentMergeRequest` - Department merger operations

### 3. Attendance Service Types (`attendanceService.types.ts`)

**Core Types:**
- `AttendanceFilter` - Attendance record filtering
- `AttendanceStats` - Attendance statistics and rates
- `DailyAttendanceReport` - Daily attendance summaries
- `WeeklyAttendanceReport` - Weekly attendance analysis
- `MonthlyAttendanceReport` - Monthly attendance reports

**Advanced Types:**
- `AttendanceTimesheet` - Employee timesheet generation
- `AttendancePattern` - Pattern analysis for irregular attendance
- `AttendanceAlert` - Alert system for attendance issues
- `AttendanceGeolocation` - Location-based attendance tracking
- `AttendanceValidationRule` - Attendance validation rules

### 4. Leave Service Types (`leaveService.types.ts`)

**Core Types:**
- `LeaveFilter` - Leave request filtering options
- `LeaveStats` - Leave utilization statistics
- `LeaveBalance` - Employee leave balance tracking
- `LeavePolicy` - Leave policy definitions
- `LeaveCalendar` - Calendar view of leaves and holidays

**Advanced Types:**
- `LeaveConflict` - Leave conflict detection
- `LeaveReport` - Comprehensive leave reports
- `LeaveNotification` - Leave-related notifications
- `LeaveEscalation` - Leave approval escalation
- `LeaveRecurring` - Recurring leave patterns
- `LeaveAdjustment` - Leave balance adjustments

### 5. Payroll Service Types (`payrollService.types.ts`)

**Core Types:**
- `PayrollFilter` - Payroll record filtering
- `PayrollStats` - Payroll statistics and summaries
- `PayrollComponent` - Salary components configuration
- `PayrollCalculation` - Detailed payroll calculations
- `PayrollSlip` - Generated payslip structure

**Advanced Types:**
- `PayrollTax` - Tax calculation configurations
- `PayrollReport` - Comprehensive payroll reports
- `PayrollAuditLog` - Payroll change tracking
- `PayrollConfiguration` - System-wide payroll settings
- `PayrollApproval` - Payroll approval workflow
- `PayrollReconciliation` - Bank reconciliation process

### 6. User Service Types (`userService.types.ts`)

**Core Types:**
- `UserProfile` - Extended user information
- `UserPreferences` - User preference settings
- `UserPermission` - Permission system structure
- `UserStats` - User system statistics
- `UserRole` - Role-based access control

**Advanced Types:**
- `UserSession` - Session management
- `UserActivityLog` - User activity tracking
- `UserInvitation` - User invitation system
- `UserSecuritySettings` - Security configurations
- `UserNotificationSettings` - Notification preferences

### 7. Reports Service Types (`reportsService.types.ts`)

**Core Report Types:**
- `EmployeeReportData` - Employee analytics reports
- `AttendanceReportData` - Attendance analysis reports
- `LeaveReportData` - Leave utilization reports
- `PayrollReportData` - Payroll cost analysis
- `DepartmentReportData` - Department performance reports

**Advanced Types:**
- `ReportTemplate` - Reusable report templates
- `ScheduledReport` - Automated report generation
- `ReportAnalytics` - Report usage analytics
- `CustomReportConfig` - Custom report builder
- `ReportValidation` - Report data validation

### 8. Notification Service Types (`notificationService.types.ts`)

**Core Types:**
- `NotificationBase` - Base notification structure
- `NotificationChannel` - Delivery channel types
- `NotificationPreferences` - User notification settings
- `NotificationStats` - Notification system metrics

**Advanced Types:**
- `NotificationTemplate` - Notification templates
- `NotificationRule` - Automated notification rules
- `NotificationEvent` - System event triggers
- `NotificationDigest` - Notification summaries
- `NotificationCampaign` - Mass notification campaigns
- `NotificationAnalytics` - Delivery and engagement analytics

### 9. Dashboard Service Types (`dashboardService.types.ts`)

**Core Types:**
- `DashboardWidget` - Widget configuration
- `DashboardLayout` - Dashboard layout management
- `ExtendedDashboardStats` - Enhanced dashboard metrics
- `DashboardMetrics` - Comprehensive system metrics

**Advanced Types:**
- `EmployeeMetrics` - Employee-focused metrics
- `AttendanceMetrics` - Attendance analytics
- `LeaveMetrics` - Leave utilization metrics
- `PayrollMetrics` - Payroll cost metrics
- `TimeSeries` - Time-based data visualization
- `Forecast` - Predictive analytics data

## Common Service Types

### Response Wrappers
- `ServiceResponse<T>` - Standard service response wrapper
- `PaginatedResponse<T>` - Paginated data response
- `BulkOperationResponse` - Bulk operation results

### Error Handling
- `ValidationError` - Field validation errors
- `ApiError` - API-level error information

### Filtering & Pagination
- `BaseFilter` - Common filtering options
- `DateRangeFilter` - Date range filtering
- `NumericRangeFilter` - Numeric range filtering

## API Types (`api.types.ts`)

### HTTP Communication
- `ApiResponse<T>` - Standard API response structure
- `ApiRequestConfig` - Request configuration options
- `ResponseMetadata` - Response metadata information
- `PaginationMetadata` - Pagination information
- `ApiError` - API error structure

### Authentication
- `AuthRequest` - Login request structure
- `AuthResponse` - Authentication response
- `RefreshTokenRequest` - Token refresh request
- `RefreshTokenResponse` - Token refresh response

### File Operations
- `FileUploadRequest` - File upload structure
- `FileUploadResponse` - Upload response information
- `ExportRequest` - Data export requests
- `ExportResponse` - Export operation results
- `ImportRequest` - Data import requests
- `ImportResponse` - Import operation results

### Advanced Features
- `SearchRequest` - Search functionality
- `SearchResponse<T>` - Search results structure
- `BulkRequest<T>` - Bulk operations request
- `BulkResponse` - Bulk operations response
- `WebSocketMessage` - Real-time messaging
- `WebhookPayload` - Webhook event structure
- `HealthCheckResponse` - System health monitoring

## Service Interfaces (`serviceInterfaces.ts`)

Defines comprehensive interfaces for each service:

- `IEmployeeService` - Employee service contract
- `IDepartmentService` - Department service contract
- `IAttendanceService` - Attendance service contract
- `ILeaveService` - Leave service contract
- `IPayrollService` - Payroll service contract
- `IUserService` - User service contract
- `IReportsService` - Reports service contract
- `INotificationService` - Notification service contract
- `IDashboardService` - Dashboard service contract

Each interface defines:
- CRUD operations
- Business-specific operations
- Statistics and analytics methods
- Validation methods
- Bulk operations where applicable

## Usage Examples

### Using Employee Service Types
```typescript
import { EmployeeFilter, EmployeeListResponse, IEmployeeService } from '@/types';

const employeeService: IEmployeeService = new EmployeeService();

const filter: EmployeeFilter = {
  department: 'Engineering',
  status: 'active',
  salaryRange: { min: 50000, max: 150000 }
};

const response: EmployeeListResponse = await employeeService.getAllEmployees(filter);
```

### Using API Types
```typescript
import { ApiResponse, EmployeeCreateRequest } from '@/types';

const createEmployee = async (employee: EmployeeCreateRequest): Promise<ApiResponse<Employee>> => {
  return await api.post('/employees', employee);
};
```

### Using Notification Types
```typescript
import { NotificationCreateRequest, NotificationChannel } from '@/types';

const notification: NotificationCreateRequest = {
  type: 'info',
  category: 'leave',
  priority: 'medium',
  title: 'Leave Request Approved',
  message: 'Your vacation leave has been approved.',
  userId: 'emp-123',
  channels: ['in_app', 'email']
};
```

## Benefits of This Type System

1. **Type Safety** - Compile-time error checking for all operations
2. **API Contract** - Clear contracts between frontend and backend
3. **Documentation** - Self-documenting code through comprehensive types
4. **Maintainability** - Easy to refactor and maintain with proper typing
5. **Developer Experience** - Enhanced IntelliSense and autocompletion
6. **Validation** - Runtime validation support through type definitions
7. **Testing** - Better test coverage with typed mock objects
8. **Consistency** - Consistent data structures across the application

This comprehensive type system ensures robust, maintainable, and scalable code for the HR Management System.