import React, { useEffect } from 'react';
import { BrowserRouter as Router, Routes, Route, Navigate } from 'react-router-dom';
import { ThemeProvider } from '@mui/material/styles';
import CssBaseline from '@mui/material/CssBaseline';
import { AuthProvider, useAuth } from './contexts/AuthContext';
import { Layout } from './components/Layout/Layout';
import { Login } from './pages/Login';
import { MobileDashboard } from './pages/MobileDashboard';
import { MobileEmployees } from './pages/MobileEmployees';
import { MobileDepartments } from './pages/MobileDepartments';
import { MobileAttendance } from './pages/MobileAttendance';
import { Leaves } from './pages/Leaves';
import { Payroll } from './pages/Payroll';
import { Reports } from './pages/Reports';
import { MailSettings } from './pages/MailSettings';
import { Holidays } from './pages/Holidays';
import { UserManagement } from './pages/UserManagement';
import { Compliance } from './pages/Compliance';
import { Recruitment } from './pages/Recruitment';
import { responsiveTheme } from './theme/responsive';
import { appConfig } from './config';

interface ProtectedRouteProps {
  children: React.ReactNode;
}

const ProtectedRoute: React.FC<ProtectedRouteProps> = ({ children }) => {
  const { user, loading } = useAuth();

  if (loading) {
    return <div>Loading...</div>;
  }

  if (!user) {
    return <Navigate to="/login" replace />;
  }

  return <Layout>{children}</Layout>;
};

function App() {
  useEffect(() => {
    document.title = appConfig.app.name;
    
    if (appConfig.development.debug) {
      console.log('App Configuration:', {
        name: appConfig.app.name,
        version: appConfig.app.version,
        apiBaseUrl: appConfig.api.baseUrl,
        features: appConfig.features
      });
    }
  }, []);

  return (
    <ThemeProvider theme={responsiveTheme}>
      <CssBaseline />
      <AuthProvider>
        <Router>
          <Routes>
            <Route path="/login" element={<Login />} />
            <Route
              path="/dashboard"
              element={
                <ProtectedRoute>
                  <MobileDashboard />
                </ProtectedRoute>
              }
            />
            <Route
              path="/employees"
              element={
                <ProtectedRoute>
                  <MobileEmployees />
                </ProtectedRoute>
              }
            />
            <Route
              path="/departments"
              element={
                <ProtectedRoute>
                  <MobileDepartments />
                </ProtectedRoute>
              }
            />
            <Route
              path="/attendance"
              element={
                <ProtectedRoute>
                  <MobileAttendance />
                </ProtectedRoute>
              }
            />
            <Route
              path="/leaves"
              element={
                <ProtectedRoute>
                  <Leaves />
                </ProtectedRoute>
              }
            />
            <Route
              path="/payroll"
              element={
                <ProtectedRoute>
                  <Payroll />
                </ProtectedRoute>
              }
            />
            <Route
              path="/reports"
              element={
                <ProtectedRoute>
                  <Reports />
                </ProtectedRoute>
              }
            />
            <Route
              path="/mail-settings"
              element={
                <ProtectedRoute>
                  <MailSettings />
                </ProtectedRoute>
              }
            />
            <Route
              path="/holidays"
              element={
                <ProtectedRoute>
                  <Holidays />
                </ProtectedRoute>
              }
            />
            <Route
              path="/user-management"
              element={
                <ProtectedRoute>
                  <UserManagement />
                </ProtectedRoute>
              }
            />
            <Route
              path="/compliance"
              element={
                <ProtectedRoute>
                  <Compliance />
                </ProtectedRoute>
              }
            />
            <Route
              path="/recruitment"
              element={
                <ProtectedRoute>
                  <Recruitment />
                </ProtectedRoute>
              }
            />
            <Route path="/" element={<Navigate to="/dashboard" replace />} />
          </Routes>
        </Router>
      </AuthProvider>
    </ThemeProvider>
  );
}

export default App;
