import React from 'react';
import {
  Box,
  Card,
  CardContent,
  Typography,
  Chip,
  Avatar,
  IconButton,
  Collapse,
  useTheme,
  useMediaQuery,
  TableContainer,
  Table,
  TableHead,
  TableBody,
  TableRow,
  TableCell,
  Paper,
} from '@mui/material';
import { ExpandMore, ExpandLess } from '@mui/icons-material';

interface MobileCardProps {
  title: string;
  subtitle?: string;
  avatar?: string | React.ReactNode;
  status?: {
    label: string;
    color: 'success' | 'error' | 'warning' | 'info' | 'default';
  };
  fields: Array<{
    label: string;
    value: string | number | React.ReactNode;
  }>;
  actions?: React.ReactNode[];
  expandableContent?: React.ReactNode;
  onClick?: () => void;
}

export const MobileCard: React.FC<MobileCardProps> = ({
  title,
  subtitle,
  avatar,
  status,
  fields,
  actions,
  expandableContent,
  onClick,
}) => {
  const [expanded, setExpanded] = React.useState(false);

  return (
    <Card 
      sx={{ 
        mb: 1, 
        cursor: onClick ? 'pointer' : 'default',
        '&:hover': onClick ? { bgcolor: 'action.hover' } : {},
      }}
      onClick={onClick}
    >
      <CardContent sx={{ p: 2, '&:last-child': { pb: 2 } }}>
        <Box sx={{ display: 'flex', alignItems: 'flex-start', justifyContent: 'space-between', mb: 1 }}>
          <Box sx={{ display: 'flex', alignItems: 'center', flex: 1, mr: 1 }}>
            {avatar && (
              <Box sx={{ mr: 2 }}>
                {typeof avatar === 'string' ? (
                  <Avatar src={avatar} sx={{ width: 40, height: 40 }} />
                ) : typeof avatar === 'object' ? (
                  <Avatar sx={{ width: 40, height: 40 }}>{avatar}</Avatar>
                ) : (
                  avatar
                )}
              </Box>
            )}
            <Box sx={{ flex: 1 }}>
              <Typography variant="h6" sx={{ fontSize: '1rem', fontWeight: 600, lineHeight: 1.2 }}>
                {title}
              </Typography>
              {subtitle && (
                <Typography variant="body2" color="textSecondary" sx={{ fontSize: '0.75rem' }}>
                  {subtitle}
                </Typography>
              )}
            </Box>
          </Box>
          
          <Box sx={{ display: 'flex', alignItems: 'center', gap: 0.5 }}>
            {status && (
              <Chip
                label={status.label}
                color={status.color}
                size="small"
                sx={{ fontSize: '0.7rem', height: 24 }}
              />
            )}
            {actions && actions.map((action, index) => (
              <Box key={index} sx={{ ml: 0.5 }}>
                {action}
              </Box>
            ))}
            {expandableContent && (
              <IconButton 
                size="small" 
                onClick={(e) => {
                  e.stopPropagation();
                  setExpanded(!expanded);
                }}
              >
                {expanded ? <ExpandLess /> : <ExpandMore />}
              </IconButton>
            )}
          </Box>
        </Box>

        {/* Main Fields */}
        <Box sx={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: 1, mt: 1 }}>
          {fields.slice(0, 4).map((field, index) => (
            <Box key={index}>
              <Typography variant="caption" color="textSecondary" display="block">
                {field.label}
              </Typography>
              <Typography variant="body2" sx={{ fontSize: '0.875rem', fontWeight: 500 }}>
                {field.value}
              </Typography>
            </Box>
          ))}
        </Box>

        {/* Expandable Content */}
        {expandableContent && (
          <Collapse in={expanded} timeout="auto" unmountOnExit>
            <Box sx={{ mt: 2, pt: 2, borderTop: '1px solid', borderColor: 'divider' }}>
              {expandableContent}
            </Box>
          </Collapse>
        )}
      </CardContent>
    </Card>
  );
};

interface ResponsiveTableProps {
  columns: Array<{
    field: string;
    label: string;
    width?: string;
    render?: (value: any, row: any) => React.ReactNode;
  }>;
  data: any[];
  mobileCardConfig: {
    title: (row: any) => string;
    subtitle?: (row: any) => string;
    avatar?: (row: any) => string | React.ReactNode;
    status?: (row: any) => { label: string; color: 'success' | 'error' | 'warning' | 'info' | 'default' };
    mainFields: string[];
    expandableFields?: string[];
    actions?: (row: any) => React.ReactNode[];
  };
  onRowClick?: (row: any) => void;
}

export const ResponsiveTable: React.FC<ResponsiveTableProps> = ({
  columns,
  data,
  mobileCardConfig,
  onRowClick,
}) => {
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down('md'));

  if (isMobile) {
    return (
      <Box>
        {data.map((row, index) => {
          const mainFields = mobileCardConfig.mainFields.map(fieldName => {
            const column = columns.find(col => col.field === fieldName);
            return {
              label: column?.label || fieldName,
              value: column?.render ? column.render(row[fieldName], row) : row[fieldName],
            };
          });

          const expandableContent = mobileCardConfig.expandableFields && (
            <Box sx={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: 1 }}>
              {mobileCardConfig.expandableFields.map(fieldName => {
                const column = columns.find(col => col.field === fieldName);
                return (
                  <Box key={fieldName}>
                    <Typography variant="caption" color="textSecondary" display="block">
                      {column?.label || fieldName}
                    </Typography>
                    <Typography variant="body2" sx={{ fontSize: '0.875rem' }}>
                      {column?.render ? column.render(row[fieldName], row) : row[fieldName]}
                    </Typography>
                  </Box>
                );
              })}
            </Box>
          );

          return (
            <MobileCard
              key={index}
              title={mobileCardConfig.title(row)}
              subtitle={mobileCardConfig.subtitle?.(row)}
              avatar={mobileCardConfig.avatar?.(row)}
              status={mobileCardConfig.status?.(row)}
              fields={mainFields}
              actions={mobileCardConfig.actions?.(row)}
              expandableContent={expandableContent}
              onClick={onRowClick ? () => onRowClick(row) : undefined}
            />
          );
        })}
      </Box>
    );
  }

  // Desktop table view
  return (
    <TableContainer component={Paper}>
      <Table>
        <TableHead>
          <TableRow>
            {columns.map((column) => (
              <TableCell key={column.field} sx={{ fontWeight: 600 }}>
                {column.label}
              </TableCell>
            ))}
          </TableRow>
        </TableHead>
        <TableBody>
          {data.map((row, index) => (
            <TableRow 
              key={index} 
              hover 
              onClick={onRowClick ? () => onRowClick(row) : undefined}
              sx={{ cursor: onRowClick ? 'pointer' : 'default' }}
            >
              {columns.map((column) => (
                <TableCell key={column.field}>
                  {column.render ? column.render(row[column.field], row) : row[column.field]}
                </TableCell>
              ))}
            </TableRow>
          ))}
        </TableBody>
      </Table>
    </TableContainer>
  );
};

interface MobileStatsGridProps {
  stats: Array<{
    title: string;
    value: string | number;
    icon?: React.ReactNode;
    color?: string;
    change?: string;
  }>;
  columns?: { xs: number; sm: number; md: number };
}

export const MobileStatsGrid: React.FC<MobileStatsGridProps> = ({ 
  stats, 
  columns = { xs: 6, sm: 4, md: 3 } 
}) => {
  return (
    <Box sx={{ 
      display: 'grid', 
      gridTemplateColumns: {
        xs: `repeat(${columns.xs}, 1fr)`,
        sm: `repeat(${columns.sm}, 1fr)`,
        md: `repeat(${columns.md}, 1fr)`,
      },
      gap: { xs: 1, sm: 2 },
      mb: 2,
    }}>
      {stats.map((stat, index) => (
        <Card key={index} sx={{ textAlign: 'center' }}>
          <CardContent sx={{ p: { xs: 1.5, sm: 2 }, '&:last-child': { pb: { xs: 1.5, sm: 2 } } }}>
            {stat.icon && (
              <Box sx={{ mb: 1, color: stat.color || 'primary.main' }}>
                {stat.icon}
              </Box>
            )}
            <Typography variant="h5" sx={{ 
              fontSize: { xs: '1.25rem', sm: '1.5rem' },
              fontWeight: 700,
              color: stat.color || 'primary.main'
            }}>
              {stat.value}
            </Typography>
            <Typography variant="caption" color="textSecondary" sx={{ 
              fontSize: { xs: '0.7rem', sm: '0.75rem' },
              display: 'block',
              lineHeight: 1.2
            }}>
              {stat.title}
            </Typography>
            {stat.change && (
              <Typography variant="caption" color="success.main" sx={{ 
                fontSize: '0.7rem',
                display: 'block',
                mt: 0.5
              }}>
                {stat.change}
              </Typography>
            )}
          </CardContent>
        </Card>
      ))}
    </Box>
  );
};