import React, { useState, useEffect } from 'react';
import {
  Card,
  CardContent,
  CardHeader,
  Typography,
  Box,
  Chip,
  List,
  ListItem,
  ListItemText,
  ListItemSecondaryAction,
  IconButton,
  Badge,
  LinearProgress,
  Button,
  Divider,
  Alert,
} from '@mui/material';
import {
  Email,
  Send,
  Inbox,
  Error,
  Schedule,
  CheckCircle,
  Settings,
  Refresh,
  MoreVert,
} from '@mui/icons-material';
import { 
  SentEmail, 
  ReceivedEmail, 
  EmailQueue,
  MailSettings as IMailSettings 
} from '../../types/services/mailService.types';
import { mailService } from '../../services';
import { useNavigate } from 'react-router-dom';

export const MailDashboard: React.FC = () => {
  const navigate = useNavigate();
  const [loading, setLoading] = useState(false);
  const [mailStats, setMailStats] = useState({
    sentToday: 0,
    receivedToday: 0,
    unreadCount: 0,
    failedCount: 0,
    queuedCount: 0,
  });
  const [recentSent, setRecentSent] = useState<SentEmail[]>([]);
  const [recentReceived, setRecentReceived] = useState<ReceivedEmail[]>([]);
  const [mailSettings, setMailSettings] = useState<IMailSettings[]>([]);
  const [connectionStatus, setConnectionStatus] = useState<'connected' | 'disconnected' | 'unknown'>('unknown');

  useEffect(() => {
    loadDashboardData();
  }, []);

  const loadDashboardData = async () => {
    setLoading(true);
    try {
      const [sentRes, receivedRes, settingsRes] = await Promise.all([
        mailService.getSentEmails(1, 5),
        mailService.getReceivedEmails(1, 5),
        mailService.getMailSettings(),
      ]);

      if (sentRes.success && sentRes.data) {
        setRecentSent(sentRes.data.emails);
        // Calculate stats from recent emails
        const today = new Date().toDateString();
        const sentToday = sentRes.data.emails.filter(
          email => email.sentAt.toDateString() === today
        ).length;
        setMailStats(prev => ({ ...prev, sentToday }));
      }

      if (receivedRes.success && receivedRes.data) {
        setRecentReceived(receivedRes.data.emails);
        const today = new Date().toDateString();
        const receivedToday = receivedRes.data.emails.filter(
          email => email.receivedAt.toDateString() === today
        ).length;
        const unreadCount = receivedRes.data.emails.filter(email => !email.isRead).length;
        setMailStats(prev => ({ ...prev, receivedToday, unreadCount }));
      }

      if (settingsRes.success && settingsRes.data) {
        setMailSettings(settingsRes.data);
        const activeSettings = settingsRes.data.find(s => s.isActive && s.isDefault);
        if (activeSettings) {
          setConnectionStatus('connected');
        } else {
          setConnectionStatus('disconnected');
        }
      }
    } catch (error) {
      console.error('Failed to load mail dashboard data:', error);
      setConnectionStatus('disconnected');
    } finally {
      setLoading(false);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'sent':
      case 'delivered':
        return 'success';
      case 'failed':
      case 'bounced':
        return 'error';
      case 'pending':
        return 'warning';
      default:
        return 'default';
    }
  };

  const StatCard: React.FC<{ 
    title: string; 
    value: number; 
    icon: React.ReactNode; 
    color?: string;
    subtitle?: string;
  }> = ({ title, value, icon, color = 'primary', subtitle }) => (
    <Card>
      <CardContent>
        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'flex-start' }}>
          <Box>
            <Typography color="textSecondary" gutterBottom>
              {title}
            </Typography>
            <Typography variant="h4" component="div">
              {value}
            </Typography>
            {subtitle && (
              <Typography variant="caption" color="textSecondary">
                {subtitle}
              </Typography>
            )}
          </Box>
          <Box sx={{ color: `${color}.main` }}>
            {icon}
          </Box>
        </Box>
      </CardContent>
    </Card>
  );

  return (
    <Box>
      {loading && <LinearProgress sx={{ mb: 2 }} />}
      
      {/* Connection Status */}
      {connectionStatus === 'disconnected' && (
        <Alert 
          severity="warning" 
          sx={{ mb: 2 }}
          action={
            <Button 
              color="inherit" 
              size="small" 
              onClick={() => navigate('/mail-settings')}
            >
              Configure
            </Button>
          }
        >
          No active mail configuration found. Please set up SMTP/IMAP settings.
        </Alert>
      )}

      {/* Mail Statistics */}
      <Box sx={{ display: 'flex', flexWrap: 'wrap', gap: 2, mb: 3 }}>
        <Box sx={{ flex: '1 1 250px', minWidth: '250px' }}>
          <StatCard
            title="Sent Today"
            value={mailStats.sentToday}
            icon={<Send />}
            color="primary"
          />
        </Box>
        <Box sx={{ flex: '1 1 250px', minWidth: '250px' }}>
          <StatCard
            title="Received Today"
            value={mailStats.receivedToday}
            icon={<Inbox />}
            color="info"
          />
        </Box>
        <Box sx={{ flex: '1 1 250px', minWidth: '250px' }}>
          <StatCard
            title="Unread"
            value={mailStats.unreadCount}
            icon={
              <Badge badgeContent={mailStats.unreadCount} color="error">
                <Email />
              </Badge>
            }
            color="warning"
          />
        </Box>
        <Box sx={{ flex: '1 1 250px', minWidth: '250px' }}>
          <StatCard
            title="Failed/Queued"
            value={mailStats.failedCount + mailStats.queuedCount}
            icon={<Error />}
            color="error"
          />
        </Box>
      </Box>

      <Box sx={{ display: 'flex', flexWrap: 'wrap', gap: 3 }}>
        {/* Recent Sent Emails */}
        <Box sx={{ flex: '1 1 400px', minWidth: '400px' }}>
          <Card>
            <CardHeader
              title="Recent Sent Emails"
              action={
                <IconButton onClick={loadDashboardData}>
                  <Refresh />
                </IconButton>
              }
            />
            <CardContent>
              {recentSent.length > 0 ? (
                <List dense>
                  {recentSent.slice(0, 5).map((email) => (
                    <React.Fragment key={email.id}>
                      <ListItem>
                        <ListItemText
                          primary={email.subject}
                          secondary={`To: ${email.to.join(', ')} • ${email.sentAt.toLocaleString()}`}
                        />
                        <ListItemSecondaryAction>
                          <Chip 
                            label={email.status} 
                            size="small" 
                            color={getStatusColor(email.status) as any}
                          />
                        </ListItemSecondaryAction>
                      </ListItem>
                      <Divider />
                    </React.Fragment>
                  ))}
                </List>
              ) : (
                <Typography variant="body2" color="textSecondary" textAlign="center" py={4}>
                  No sent emails found
                </Typography>
              )}
              <Box sx={{ mt: 2, textAlign: 'center' }}>
                <Button 
                  size="small" 
                  onClick={() => navigate('/mail-settings')}
                >
                  View All Sent
                </Button>
              </Box>
            </CardContent>
          </Card>
        </Box>

        {/* Recent Received Emails */}
        <Box sx={{ flex: '1 1 400px', minWidth: '400px' }}>
          <Card>
            <CardHeader
              title="Recent Received Emails"
              action={
                <IconButton onClick={loadDashboardData}>
                  <Refresh />
                </IconButton>
              }
            />
            <CardContent>
              {recentReceived.length > 0 ? (
                <List dense>
                  {recentReceived.slice(0, 5).map((email) => (
                    <React.Fragment key={email.id}>
                      <ListItem>
                        <ListItemText
                          primary={
                            <Box sx={{ display: 'flex', alignItems: 'center' }}>
                              <Typography 
                                variant="body2" 
                                sx={{ 
                                  fontWeight: email.isRead ? 'normal' : 'bold',
                                  flex: 1 
                                }}
                              >
                                {email.subject}
                              </Typography>
                              {!email.isRead && (
                                <Chip label="New" size="small" color="primary" />
                              )}
                            </Box>
                          }
                          secondary={`From: ${email.from} • ${email.receivedAt.toLocaleString()}`}
                        />
                      </ListItem>
                      <Divider />
                    </React.Fragment>
                  ))}
                </List>
              ) : (
                <Typography variant="body2" color="textSecondary" textAlign="center" py={4}>
                  No received emails found
                </Typography>
              )}
              <Box sx={{ mt: 2, textAlign: 'center' }}>
                <Button 
                  size="small" 
                  onClick={() => navigate('/mail-settings')}
                >
                  View All Received
                </Button>
              </Box>
            </CardContent>
          </Card>
        </Box>
      </Box>

        {/* Mail Configuration Status */}
        <Box sx={{ mt: 3 }}>
          <Card>
            <CardHeader
              title="Mail Configuration"
              action={
                <Button
                  startIcon={<Settings />}
                  onClick={() => navigate('/mail-settings')}
                >
                  Manage Settings
                </Button>
              }
            />
            <CardContent>
                {mailSettings.length > 0 ? (
                  <Box sx={{ display: 'flex', flexWrap: 'wrap', gap: 2 }}>
                    {mailSettings.map((setting) => (
                      <Box key={setting.id} sx={{ flex: '1 1 300px', minWidth: '300px' }}>
                        <Card variant="outlined">
                        <CardContent>
                          <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'flex-start' }}>
                            <Box>
                              <Typography variant="h6">{setting.name}</Typography>
                              <Typography variant="body2" color="text.secondary">
                                {setting.defaultFrom}
                              </Typography>
                              <Box sx={{ mt: 1 }}>
                                {setting.isDefault && (
                                  <Chip label="Default" size="small" color="primary" sx={{ mr: 1 }} />
                                )}
                                <Chip 
                                  label={setting.isActive ? "Active" : "Inactive"} 
                                  size="small" 
                                  color={setting.isActive ? "success" : "default"} 
                                />
                              </Box>
                            </Box>
                            <Box>
                              {setting.isActive ? (
                                <CheckCircle color="success" />
                              ) : (
                                <Schedule color="disabled" />
                              )}
                            </Box>
                          </Box>
                        </CardContent>
                        </Card>
                      </Box>
                    ))}
                  </Box>
                ) : (
                    <Typography variant="body2" color="textSecondary" textAlign="center">
                      No mail configurations found. Click "Manage Settings" to add one.
                    </Typography>
                )}
            </CardContent>
          </Card>
        </Box>
    </Box>
  );
};