import React, { useState } from 'react';
import {
  IconButton,
  Menu,
  MenuItem,
  ListItemIcon,
  ListItemText,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  TextField,
  Button,
  Box,
  Select,
  FormControl,
  InputLabel,
  Alert,
  CircularProgress,
} from '@mui/material';
import {
  Email,
  Send,
  Close,
} from '@mui/icons-material';
import { mailService } from '../../services';
import { EmailTemplate } from '../../types/services/mailService.types';

interface QuickEmailButtonProps {
  recipientEmail?: string;
  recipientName?: string;
  prefilledSubject?: string;
  prefilledContent?: string;
  templates?: EmailTemplate[];
  size?: 'small' | 'medium' | 'large';
  variant?: 'contained' | 'outlined' | 'text';
  onEmailSent?: () => void;
}

export const QuickEmailButton: React.FC<QuickEmailButtonProps> = ({
  recipientEmail = '',
  recipientName = '',
  prefilledSubject = '',
  prefilledContent = '',
  templates = [],
  size = 'medium',
  variant = 'text',
  onEmailSent,
}) => {
  const [anchorEl, setAnchorEl] = useState<null | HTMLElement>(null);
  const [emailDialog, setEmailDialog] = useState(false);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');
  const [emailForm, setEmailForm] = useState({
    to: recipientEmail,
    subject: prefilledSubject,
    content: prefilledContent,
    templateId: '',
  });
  const [availableTemplates, setAvailableTemplates] = useState<EmailTemplate[]>(templates);

  React.useEffect(() => {
    if (templates.length === 0) {
      loadTemplates();
    }
  }, []);

  React.useEffect(() => {
    setEmailForm({
      to: recipientEmail,
      subject: prefilledSubject,
      content: prefilledContent,
      templateId: '',
    });
  }, [recipientEmail, prefilledSubject, prefilledContent]);

  const loadTemplates = async () => {
    try {
      const result = await mailService.getEmailTemplates();
      if (result.success && result.data) {
        setAvailableTemplates(result.data);
      }
    } catch (error) {
      console.error('Failed to load email templates:', error);
    }
  };

  const handleMenuClick = (event: React.MouseEvent<HTMLElement>) => {
    setAnchorEl(event.currentTarget);
  };

  const handleMenuClose = () => {
    setAnchorEl(null);
  };

  const handleQuickEmail = (templateId?: string) => {
    if (templateId) {
      const template = availableTemplates.find(t => t.id === templateId);
      if (template) {
        setEmailForm({
          to: recipientEmail,
          subject: template.subject.replace('{{employeeName}}', recipientName),
          content: template.textContent.replace('{{employeeName}}', recipientName),
          templateId: template.id,
        });
      }
    }
    setEmailDialog(true);
    handleMenuClose();
  };

  const handleSendEmail = async () => {
    if (!emailForm.to || !emailForm.subject) {
      setError('Please fill in required fields (To, Subject)');
      return;
    }

    setLoading(true);
    setError('');

    try {
      const message = {
        to: emailForm.to,
        subject: emailForm.subject,
        html: `<p>${emailForm.content.replace(/\n/g, '<br>')}</p>`,
        text: emailForm.content,
        templateId: emailForm.templateId || undefined,
      };

      const result = await mailService.sendEmail(message);
      if (result.success) {
        setEmailDialog(false);
        setEmailForm({
          to: recipientEmail,
          subject: prefilledSubject,
          content: prefilledContent,
          templateId: '',
        });
        if (onEmailSent) {
          onEmailSent();
        }
      } else {
        setError(result.error || 'Failed to send email');
      }
    } catch (error) {
      setError('Failed to send email');
    } finally {
      setLoading(false);
    }
  };

  const handleTemplateChange = (templateId: string) => {
    const template = availableTemplates.find(t => t.id === templateId);
    if (template) {
      setEmailForm({
        ...emailForm,
        subject: template.subject.replace('{{employeeName}}', recipientName),
        content: template.textContent.replace('{{employeeName}}', recipientName),
        templateId: template.id,
      });
    } else {
      setEmailForm({
        ...emailForm,
        subject: prefilledSubject,
        content: prefilledContent,
        templateId: '',
      });
    }
  };

  return (
    <>
      <IconButton
        size={size}
        onClick={handleMenuClick}
        color="primary"
        title="Send Email"
      >
        <Email />
      </IconButton>

      <Menu
        anchorEl={anchorEl}
        open={Boolean(anchorEl)}
        onClose={handleMenuClose}
        PaperProps={{
          sx: { minWidth: 200 }
        }}
      >
        <MenuItem onClick={() => handleQuickEmail()}>
          <ListItemIcon>
            <Send fontSize="small" />
          </ListItemIcon>
          <ListItemText>Send Custom Email</ListItemText>
        </MenuItem>
        
        {availableTemplates
          .filter(t => t.isActive)
          .slice(0, 3)
          .map((template) => (
            <MenuItem 
              key={template.id} 
              onClick={() => handleQuickEmail(template.id)}
            >
              <ListItemIcon>
                <Email fontSize="small" />
              </ListItemIcon>
              <ListItemText>{template.name}</ListItemText>
            </MenuItem>
          ))}
      </Menu>

      <Dialog
        open={emailDialog}
        onClose={() => setEmailDialog(false)}
        maxWidth="md"
        fullWidth
      >
        <DialogTitle sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
          Send Email
          <IconButton onClick={() => setEmailDialog(false)}>
            <Close />
          </IconButton>
        </DialogTitle>

        <DialogContent>
          {error && (
            <Alert severity="error" sx={{ mb: 2 }}>
              {error}
            </Alert>
          )}

          <Box sx={{ display: 'flex', flexDirection: 'column', gap: 2 }}>
            <TextField
              label="To"
              value={emailForm.to}
              onChange={(e) => setEmailForm({ ...emailForm, to: e.target.value })}
              fullWidth
              required
              type="email"
            />
            
            <Box sx={{ display: 'flex', gap: 2, flexWrap: 'wrap' }}>
              <Box sx={{ flex: '2 1 300px' }}>
                <TextField
                  label="Subject"
                  value={emailForm.subject}
                  onChange={(e) => setEmailForm({ ...emailForm, subject: e.target.value })}
                  fullWidth
                  required
                />
              </Box>
              
              <Box sx={{ flex: '1 1 200px' }}>
                <FormControl fullWidth>
                  <InputLabel>Template</InputLabel>
                  <Select
                    value={emailForm.templateId}
                    onChange={(e) => handleTemplateChange(e.target.value)}
                  >
                    <MenuItem value="">None</MenuItem>
                    {availableTemplates
                      .filter(t => t.isActive)
                      .map((template) => (
                        <MenuItem key={template.id} value={template.id}>
                          {template.name}
                        </MenuItem>
                      ))}
                  </Select>
                </FormControl>
              </Box>
            </Box>
            
            <TextField
              label="Message"
              value={emailForm.content}
              onChange={(e) => setEmailForm({ ...emailForm, content: e.target.value })}
              multiline
              rows={8}
              fullWidth
            />
          </Box>
        </DialogContent>

        <DialogActions>
          <Button onClick={() => setEmailDialog(false)}>
            Cancel
          </Button>
          <Button
            variant="contained"
            startIcon={loading ? <CircularProgress size={16} /> : <Send />}
            onClick={handleSendEmail}
            disabled={loading || !emailForm.to || !emailForm.subject}
          >
            {loading ? 'Sending...' : 'Send Email'}
          </Button>
        </DialogActions>
      </Dialog>
    </>
  );
};