import React, { useState, useEffect } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  TextField,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Box,
  Chip,
  Typography,
  IconButton,
  Stepper,
  Step,
  StepLabel
} from '@mui/material';
import { Add, Close, CloudUpload } from '@mui/icons-material';
import { Candidate, CandidateRequest } from '../../types/recruitment';
import { LoaderOverlay } from '../Common/LoaderOverlay';

interface CandidateDialogProps {
  open: boolean;
  onClose: () => void;
  onSubmit: (data: CandidateRequest) => Promise<void>;
  editData?: Candidate;
  jobId?: string;
}

export const CandidateDialog: React.FC<CandidateDialogProps> = ({
  open,
  onClose,
  onSubmit,
  editData,
  jobId
}) => {
  const [loading, setLoading] = useState(false);
  const [activeStep, setActiveStep] = useState(0);
  const [formData, setFormData] = useState<CandidateRequest>({
    firstName: '',
    lastName: '',
    email: '',
    phone: '',
    address: '',
    linkedInProfile: '',
    portfolioUrl: '',
    currentPosition: '',
    currentCompany: '',
    totalExperience: 0,
    expectedSalary: 0,
    noticePeriod: '',
    skills: [],
    jobId: jobId || '',
    source: 'Website'
  });

  const [newSkill, setNewSkill] = useState('');

  const steps = ['Basic Information', 'Professional Details', 'Skills & Experience'];

  useEffect(() => {
    if (editData) {
      setFormData({
        firstName: editData.firstName,
        lastName: editData.lastName,
        email: editData.email,
        phone: editData.phone,
        address: editData.address,
        linkedInProfile: editData.linkedInProfile,
        portfolioUrl: editData.portfolioUrl,
        currentPosition: editData.currentPosition,
        currentCompany: editData.currentCompany,
        totalExperience: editData.totalExperience,
        expectedSalary: editData.expectedSalary,
        noticePeriod: editData.noticePeriod,
        skills: editData.skills,
        source: editData.source as any
      });
    } else {
      setFormData({
        firstName: '',
        lastName: '',
        email: '',
        phone: '',
        address: '',
        linkedInProfile: '',
        portfolioUrl: '',
        currentPosition: '',
        currentCompany: '',
        totalExperience: 0,
        expectedSalary: 0,
        noticePeriod: '',
        skills: [],
        jobId: jobId || '',
        source: 'Website'
      });
    }
    setActiveStep(0);
  }, [editData, open, jobId]);

  const handleNext = () => {
    setActiveStep((prevActiveStep) => prevActiveStep + 1);
  };

  const handleBack = () => {
    setActiveStep((prevActiveStep) => prevActiveStep - 1);
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    try {
      await onSubmit(formData);
      onClose();
    } catch (error) {
      console.error('Failed to save candidate:', error);
    } finally {
      setLoading(false);
    }
  };

  const addSkill = () => {
    if (newSkill.trim()) {
      setFormData(prev => ({
        ...prev,
        skills: [...prev.skills, newSkill.trim()]
      }));
      setNewSkill('');
    }
  };

  const removeSkill = (index: number) => {
    setFormData(prev => ({
      ...prev,
      skills: prev.skills.filter((_, i) => i !== index)
    }));
  };

  const isStepValid = (step: number) => {
    switch (step) {
      case 0:
        return formData.firstName && formData.lastName && formData.email && formData.phone;
      case 1:
        return formData.totalExperience >= 0;
      case 2:
        return true;
      default:
        return false;
    }
  };

  const renderStepContent = (step: number) => {
    switch (step) {
      case 0:
        return (
          <Box sx={{ display: 'grid', gap: 3, pt: 2 }}>
            <Box sx={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: 2 }}>
              <TextField
                label="First Name"
                value={formData.firstName}
                onChange={(e) => setFormData(prev => ({ ...prev, firstName: e.target.value }))}
                required
              />
              <TextField
                label="Last Name"
                value={formData.lastName}
                onChange={(e) => setFormData(prev => ({ ...prev, lastName: e.target.value }))}
                required
              />
            </Box>

            <TextField
              label="Email"
              type="email"
              value={formData.email}
              onChange={(e) => setFormData(prev => ({ ...prev, email: e.target.value }))}
              required
            />

            <TextField
              label="Phone Number"
              value={formData.phone}
              onChange={(e) => setFormData(prev => ({ ...prev, phone: e.target.value }))}
              required
            />

            <TextField
              label="Address"
              value={formData.address}
              onChange={(e) => setFormData(prev => ({ ...prev, address: e.target.value }))}
              multiline
              rows={2}
            />

            <FormControl required>
              <InputLabel>Source</InputLabel>
              <Select
                value={formData.source}
                onChange={(e) => setFormData(prev => ({ ...prev, source: e.target.value }))}
              >
                <MenuItem value="Website">Company Website</MenuItem>
                <MenuItem value="LinkedIn">LinkedIn</MenuItem>
                <MenuItem value="Referral">Employee Referral</MenuItem>
                <MenuItem value="Job Board">Job Board</MenuItem>
                <MenuItem value="Agency">Recruitment Agency</MenuItem>
                <MenuItem value="Direct">Direct Application</MenuItem>
              </Select>
            </FormControl>
          </Box>
        );

      case 1:
        return (
          <Box sx={{ display: 'grid', gap: 3, pt: 2 }}>
            <TextField
              label="LinkedIn Profile"
              value={formData.linkedInProfile}
              onChange={(e) => setFormData(prev => ({ ...prev, linkedInProfile: e.target.value }))}
              placeholder="https://linkedin.com/in/username"
            />

            <TextField
              label="Portfolio URL"
              value={formData.portfolioUrl}
              onChange={(e) => setFormData(prev => ({ ...prev, portfolioUrl: e.target.value }))}
              placeholder="https://portfolio.com"
            />

            <Box sx={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: 2 }}>
              <TextField
                label="Current Position"
                value={formData.currentPosition}
                onChange={(e) => setFormData(prev => ({ ...prev, currentPosition: e.target.value }))}
              />
              <TextField
                label="Current Company"
                value={formData.currentCompany}
                onChange={(e) => setFormData(prev => ({ ...prev, currentCompany: e.target.value }))}
              />
            </Box>

            <Box sx={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: 2 }}>
              <TextField
                label="Total Experience (Years)"
                type="number"
                value={formData.totalExperience}
                onChange={(e) => setFormData(prev => ({ ...prev, totalExperience: parseFloat(e.target.value) || 0 }))}
                required
              />
              <TextField
                label="Expected Salary (₹)"
                type="number"
                value={formData.expectedSalary}
                onChange={(e) => setFormData(prev => ({ ...prev, expectedSalary: parseInt(e.target.value) || 0 }))}
              />
            </Box>

            <TextField
              label="Notice Period"
              value={formData.noticePeriod}
              onChange={(e) => setFormData(prev => ({ ...prev, noticePeriod: e.target.value }))}
              placeholder="e.g., 30 days, Immediate"
            />
          </Box>
        );

      case 2:
        return (
          <Box sx={{ display: 'grid', gap: 3, pt: 2 }}>
            <Box>
              <Typography variant="subtitle2" gutterBottom>Skills</Typography>
              <Box sx={{ display: 'flex', gap: 1, mb: 2 }}>
                <TextField
                  placeholder="Add skill"
                  value={newSkill}
                  onChange={(e) => setNewSkill(e.target.value)}
                  size="small"
                  sx={{ flexGrow: 1 }}
                  onKeyPress={(e) => e.key === 'Enter' && addSkill()}
                />
                <IconButton onClick={addSkill} color="primary">
                  <Add />
                </IconButton>
              </Box>
              <Box sx={{ display: 'flex', flexWrap: 'wrap', gap: 1 }}>
                {formData.skills.map((skill, index) => (
                  <Chip
                    key={index}
                    label={skill}
                    onDelete={() => removeSkill(index)}
                    size="small"
                    color="primary"
                    variant="outlined"
                  />
                ))}
              </Box>
            </Box>

            <Box>
              <Typography variant="subtitle2" gutterBottom>Resume Upload</Typography>
              <Button
                variant="outlined"
                component="label"
                startIcon={<CloudUpload />}
                fullWidth
                sx={{ mb: 1 }}
              >
                Upload Resume
                <input
                  type="file"
                  hidden
                  accept=".pdf,.doc,.docx"
                  onChange={(e) => {
                    const file = e.target.files?.[0];
                    if (file) {
                      setFormData(prev => ({ ...prev, resumeFile: file }));
                    }
                  }}
                />
              </Button>
              {formData.resumeFile && (
                <Typography variant="caption" color="success.main">
                  File selected: {formData.resumeFile.name}
                </Typography>
              )}
            </Box>

            <Box>
              <Typography variant="subtitle2" gutterBottom>Cover Letter (Optional)</Typography>
              <Button
                variant="outlined"
                component="label"
                startIcon={<CloudUpload />}
                fullWidth
                sx={{ mb: 1 }}
              >
                Upload Cover Letter
                <input
                  type="file"
                  hidden
                  accept=".pdf,.doc,.docx"
                  onChange={(e) => {
                    const file = e.target.files?.[0];
                    if (file) {
                      setFormData(prev => ({ ...prev, coverLetterFile: file }));
                    }
                  }}
                />
              </Button>
              {formData.coverLetterFile && (
                <Typography variant="caption" color="success.main">
                  File selected: {formData.coverLetterFile.name}
                </Typography>
              )}
            </Box>
          </Box>
        );

      default:
        return null;
    }
  };

  return (
    <Dialog open={open} onClose={onClose} maxWidth="md" fullWidth>
      <LoaderOverlay open={loading} />
      <DialogTitle>
        {editData ? 'Edit Candidate' : 'Add New Candidate'}
        {jobId && !editData && (
          <Typography variant="subtitle2" color="text.secondary">
            Applying for Job ID: {jobId}
          </Typography>
        )}
      </DialogTitle>
      
      <DialogContent>
        <Stepper activeStep={activeStep} sx={{ mb: 4 }}>
          {steps.map((label) => (
            <Step key={label}>
              <StepLabel>{label}</StepLabel>
            </Step>
          ))}
        </Stepper>

        {renderStepContent(activeStep)}
      </DialogContent>

      <DialogActions>
        <Button onClick={onClose}>Cancel</Button>
        <Box sx={{ flex: '1 1 auto' }} />
        {activeStep !== 0 && (
          <Button onClick={handleBack}>Back</Button>
        )}
        {activeStep === steps.length - 1 ? (
          <Button 
            variant="contained" 
            onClick={handleSubmit} 
            disabled={loading || !isStepValid(activeStep)}
          >
            {editData ? 'Update' : 'Add'} Candidate
          </Button>
        ) : (
          <Button
            variant="contained"
            onClick={handleNext}
            disabled={!isStepValid(activeStep)}
          >
            Next
          </Button>
        )}
      </DialogActions>
    </Dialog>
  );
};