import React, { useState, useEffect } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  TextField,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Box,
  Chip,
  Typography,
  Grid,
  IconButton
} from '@mui/material';
import { Add, Close } from '@mui/icons-material';
import { DatePicker } from '@mui/x-date-pickers/DatePicker';
import { JobPosting, JobPostingRequest } from '../../types/recruitment';
import { LoaderOverlay } from '../Common/LoaderOverlay';

interface JobPostingDialogProps {
  open: boolean;
  onClose: () => void;
  onSubmit: (data: JobPostingRequest) => Promise<void>;
  editData?: JobPosting;
}

export const JobPostingDialog: React.FC<JobPostingDialogProps> = ({
  open,
  onClose,
  onSubmit,
  editData
}) => {
  const [loading, setLoading] = useState(false);
  const [formData, setFormData] = useState<JobPostingRequest>({
    title: '',
    department: '',
    location: '',
    employmentType: 'Full-time',
    experienceLevel: 'Mid Level',
    salaryRange: { min: 0, max: 0, currency: 'INR' },
    description: '',
    requirements: [],
    responsibilities: [],
    benefits: [],
    applicationDeadline: '',
    hiringManager: ''
  });

  const [newRequirement, setNewRequirement] = useState('');
  const [newResponsibility, setNewResponsibility] = useState('');
  const [newBenefit, setNewBenefit] = useState('');

  useEffect(() => {
    if (editData) {
      setFormData({
        title: editData.title,
        department: editData.department,
        location: editData.location,
        employmentType: editData.employmentType,
        experienceLevel: editData.experienceLevel,
        salaryRange: editData.salaryRange,
        description: editData.description,
        requirements: editData.requirements,
        responsibilities: editData.responsibilities,
        benefits: editData.benefits,
        applicationDeadline: editData.applicationDeadline,
        hiringManager: editData.hiringManager
      });
    } else {
      setFormData({
        title: '',
        department: '',
        location: '',
        employmentType: 'Full-time',
        experienceLevel: 'Mid Level',
        salaryRange: { min: 0, max: 0, currency: 'INR' },
        description: '',
        requirements: [],
        responsibilities: [],
        benefits: [],
        applicationDeadline: '',
        hiringManager: ''
      });
    }
  }, [editData, open]);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    try {
      await onSubmit(formData);
      onClose();
    } catch (error) {
      console.error('Failed to save job posting:', error);
    } finally {
      setLoading(false);
    }
  };

  const addRequirement = () => {
    if (newRequirement.trim()) {
      setFormData(prev => ({
        ...prev,
        requirements: [...prev.requirements, newRequirement.trim()]
      }));
      setNewRequirement('');
    }
  };

  const addResponsibility = () => {
    if (newResponsibility.trim()) {
      setFormData(prev => ({
        ...prev,
        responsibilities: [...prev.responsibilities, newResponsibility.trim()]
      }));
      setNewResponsibility('');
    }
  };

  const addBenefit = () => {
    if (newBenefit.trim()) {
      setFormData(prev => ({
        ...prev,
        benefits: [...prev.benefits, newBenefit.trim()]
      }));
      setNewBenefit('');
    }
  };

  const removeRequirement = (index: number) => {
    setFormData(prev => ({
      ...prev,
      requirements: prev.requirements.filter((_, i) => i !== index)
    }));
  };

  const removeResponsibility = (index: number) => {
    setFormData(prev => ({
      ...prev,
      responsibilities: prev.responsibilities.filter((_, i) => i !== index)
    }));
  };

  const removeBenefit = (index: number) => {
    setFormData(prev => ({
      ...prev,
      benefits: prev.benefits.filter((_, i) => i !== index)
    }));
  };

  return (
    <Dialog open={open} onClose={onClose} maxWidth="md" fullWidth>
      <LoaderOverlay open={loading} />
      <DialogTitle>{editData ? 'Edit Job Posting' : 'Create New Job Posting'}</DialogTitle>
      
      <form onSubmit={handleSubmit}>
        <DialogContent>
          <Box sx={{ display: 'grid', gap: 3, pt: 2 }}>
            <TextField
              label="Job Title"
              value={formData.title}
              onChange={(e) => setFormData(prev => ({ ...prev, title: e.target.value }))}
              required
              fullWidth
            />

            <Box sx={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: 2 }}>
              <TextField
                label="Department"
                value={formData.department}
                onChange={(e) => setFormData(prev => ({ ...prev, department: e.target.value }))}
                required
              />
              <TextField
                label="Location"
                value={formData.location}
                onChange={(e) => setFormData(prev => ({ ...prev, location: e.target.value }))}
                required
              />
            </Box>

            <Box sx={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: 2 }}>
              <FormControl required>
                <InputLabel>Employment Type</InputLabel>
                <Select
                  value={formData.employmentType}
                  onChange={(e) => setFormData(prev => ({ ...prev, employmentType: e.target.value as any }))}
                >
                  <MenuItem value="Full-time">Full-time</MenuItem>
                  <MenuItem value="Part-time">Part-time</MenuItem>
                  <MenuItem value="Contract">Contract</MenuItem>
                  <MenuItem value="Temporary">Temporary</MenuItem>
                  <MenuItem value="Internship">Internship</MenuItem>
                </Select>
              </FormControl>

              <FormControl required>
                <InputLabel>Experience Level</InputLabel>
                <Select
                  value={formData.experienceLevel}
                  onChange={(e) => setFormData(prev => ({ ...prev, experienceLevel: e.target.value as any }))}
                >
                  <MenuItem value="Entry Level">Entry Level</MenuItem>
                  <MenuItem value="Mid Level">Mid Level</MenuItem>
                  <MenuItem value="Senior Level">Senior Level</MenuItem>
                  <MenuItem value="Executive">Executive</MenuItem>
                </Select>
              </FormControl>
            </Box>

            <Box>
              <Typography variant="subtitle2" gutterBottom>Salary Range</Typography>
              <Box sx={{ display: 'grid', gridTemplateColumns: '1fr 1fr 100px', gap: 2 }}>
                <TextField
                  label="Minimum Salary"
                  type="number"
                  value={formData.salaryRange.min}
                  onChange={(e) => setFormData(prev => ({ 
                    ...prev, 
                    salaryRange: { ...prev.salaryRange, min: parseInt(e.target.value) || 0 }
                  }))}
                />
                <TextField
                  label="Maximum Salary"
                  type="number"
                  value={formData.salaryRange.max}
                  onChange={(e) => setFormData(prev => ({ 
                    ...prev, 
                    salaryRange: { ...prev.salaryRange, max: parseInt(e.target.value) || 0 }
                  }))}
                />
                <TextField
                  label="Currency"
                  value={formData.salaryRange.currency}
                  onChange={(e) => setFormData(prev => ({ 
                    ...prev, 
                    salaryRange: { ...prev.salaryRange, currency: e.target.value }
                  }))}
                />
              </Box>
            </Box>

            <TextField
              label="Job Description"
              value={formData.description}
              onChange={(e) => setFormData(prev => ({ ...prev, description: e.target.value }))}
              multiline
              rows={4}
              required
            />

            <Box>
              <Typography variant="subtitle2" gutterBottom>Requirements</Typography>
              <Box sx={{ display: 'flex', gap: 1, mb: 1 }}>
                <TextField
                  placeholder="Add requirement"
                  value={newRequirement}
                  onChange={(e) => setNewRequirement(e.target.value)}
                  size="small"
                  sx={{ flexGrow: 1 }}
                />
                <IconButton onClick={addRequirement} color="primary">
                  <Add />
                </IconButton>
              </Box>
              <Box sx={{ display: 'flex', flexWrap: 'wrap', gap: 1 }}>
                {formData.requirements.map((req, index) => (
                  <Chip
                    key={index}
                    label={req}
                    onDelete={() => removeRequirement(index)}
                    size="small"
                  />
                ))}
              </Box>
            </Box>

            <Box>
              <Typography variant="subtitle2" gutterBottom>Responsibilities</Typography>
              <Box sx={{ display: 'flex', gap: 1, mb: 1 }}>
                <TextField
                  placeholder="Add responsibility"
                  value={newResponsibility}
                  onChange={(e) => setNewResponsibility(e.target.value)}
                  size="small"
                  sx={{ flexGrow: 1 }}
                />
                <IconButton onClick={addResponsibility} color="primary">
                  <Add />
                </IconButton>
              </Box>
              <Box sx={{ display: 'flex', flexWrap: 'wrap', gap: 1 }}>
                {formData.responsibilities.map((resp, index) => (
                  <Chip
                    key={index}
                    label={resp}
                    onDelete={() => removeResponsibility(index)}
                    size="small"
                  />
                ))}
              </Box>
            </Box>

            <Box>
              <Typography variant="subtitle2" gutterBottom>Benefits</Typography>
              <Box sx={{ display: 'flex', gap: 1, mb: 1 }}>
                <TextField
                  placeholder="Add benefit"
                  value={newBenefit}
                  onChange={(e) => setNewBenefit(e.target.value)}
                  size="small"
                  sx={{ flexGrow: 1 }}
                />
                <IconButton onClick={addBenefit} color="primary">
                  <Add />
                </IconButton>
              </Box>
              <Box sx={{ display: 'flex', flexWrap: 'wrap', gap: 1 }}>
                {formData.benefits.map((benefit, index) => (
                  <Chip
                    key={index}
                    label={benefit}
                    onDelete={() => removeBenefit(index)}
                    size="small"
                  />
                ))}
              </Box>
            </Box>

            <Box sx={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: 2 }}>
              <TextField
                label="Application Deadline"
                type="date"
                value={formData.applicationDeadline}
                onChange={(e) => setFormData(prev => ({ ...prev, applicationDeadline: e.target.value }))}
                InputLabelProps={{ shrink: true }}
                required
              />
              <TextField
                label="Hiring Manager"
                value={formData.hiringManager}
                onChange={(e) => setFormData(prev => ({ ...prev, hiringManager: e.target.value }))}
                required
              />
            </Box>
          </Box>
        </DialogContent>

        <DialogActions>
          <Button onClick={onClose}>Cancel</Button>
          <Button type="submit" variant="contained" disabled={loading}>
            {editData ? 'Update' : 'Create'} Job Posting
          </Button>
        </DialogActions>
      </form>
    </Dialog>
  );
};