import React, { useState, useEffect } from 'react';
import {
  Grid,
  Paper,
  Typography,
  Box,
  Card,
  CardContent,
  LinearProgress,
} from '@mui/material';
import {
  People,
  Business,
  Schedule,
  Payment,
  TrendingUp,
  EventNote,
} from '@mui/icons-material';
import {
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
  PieChart,
  Pie,
  Cell,
  LineChart,
  Line,
} from 'recharts';
import { DashboardStats } from '../types';

const COLORS = ['#0088FE', '#00C49F', '#FFBB28', '#FF8042'];

const mockStats: DashboardStats = {
  totalEmployees: 145,
  activeEmployees: 138,
  totalDepartments: 8,
  pendingLeaves: 12,
  todayAttendance: 92,
  monthlyPayroll: 485000,
};

const attendanceData = [
  { name: 'Mon', present: 135, absent: 10 },
  { name: 'Tue', present: 132, absent: 13 },
  { name: 'Wed', present: 140, absent: 5 },
  { name: 'Thu', present: 138, absent: 7 },
  { name: 'Fri', present: 130, absent: 15 },
];

const departmentData = [
  { name: 'Engineering', value: 45 },
  { name: 'Sales', value: 32 },
  { name: 'Marketing', value: 28 },
  { name: 'HR', value: 15 },
  { name: 'Finance', value: 25 },
];

const payrollTrend = [
  { month: 'Jan', amount: 450000 },
  { month: 'Feb', amount: 465000 },
  { month: 'Mar', amount: 470000 },
  { month: 'Apr', amount: 485000 },
  { month: 'May', amount: 480000 },
  { month: 'Jun', amount: 485000 },
];

interface StatsCardProps {
  title: string;
  value: string | number;
  icon: React.ReactNode;
  color: string;
  change?: string;
}

const StatsCard: React.FC<StatsCardProps> = ({ title, value, icon, color, change }) => (
  <Card sx={{ height: '100%' }}>
    <CardContent>
      <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
        <Box>
          <Typography color="textSecondary" gutterBottom variant="body2">
            {title}
          </Typography>
          <Typography variant="h4" component="h2">
            {value}
          </Typography>
          {change && (
            <Typography variant="body2" sx={{ color: 'success.main', mt: 1 }}>
              {change}
            </Typography>
          )}
        </Box>
        <Box
          sx={{
            backgroundColor: color,
            borderRadius: 2,
            p: 1.5,
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
          }}
        >
          {icon}
        </Box>
      </Box>
    </CardContent>
  </Card>
);

export const Dashboard: React.FC = () => {
  const [stats, setStats] = useState<DashboardStats>(mockStats);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    // Simulate API call
    setTimeout(() => {
      setStats(mockStats);
      setLoading(false);
    }, 1000);
  }, []);

  const attendanceRate = Math.round((stats.todayAttendance / stats.totalEmployees) * 100);

  if (loading) {
    return (
      <Box sx={{ width: '100%', mt: 2 }}>
        <LinearProgress />
      </Box>
    );
  }

  return (
    <Box>
      <Typography variant="h4" gutterBottom>
        Dashboard Overview
      </Typography>
      <Grid container spacing={3}>
        {/* Stats Cards */}
        <Grid
          size={{
            xs: 12,
            sm: 6,
            md: 3
          }}>
          <StatsCard
            title="Total Employees"
            value={stats.totalEmployees}
            icon={<People sx={{ color: 'white' }} />}
            color="#1976d2"
            change="+5% from last month"
          />
        </Grid>
        <Grid
          size={{
            xs: 12,
            sm: 6,
            md: 3
          }}>
          <StatsCard
            title="Active Employees"
            value={stats.activeEmployees}
            icon={<TrendingUp sx={{ color: 'white' }} />}
            color="#2e7d32"
          />
        </Grid>
        <Grid
          size={{
            xs: 12,
            sm: 6,
            md: 3
          }}>
          <StatsCard
            title="Departments"
            value={stats.totalDepartments}
            icon={<Business sx={{ color: 'white' }} />}
            color="#ed6c02"
          />
        </Grid>
        <Grid
          size={{
            xs: 12,
            sm: 6,
            md: 3
          }}>
          <StatsCard
            title="Pending Leaves"
            value={stats.pendingLeaves}
            icon={<EventNote sx={{ color: 'white' }} />}
            color="#d32f2f"
          />
        </Grid>

        {/* Attendance Rate */}
        <Grid
          size={{
            xs: 12,
            md: 6
          }}>
          <Paper sx={{ p: 3, height: '400px' }}>
            <Typography variant="h6" gutterBottom>
              Weekly Attendance
            </Typography>
            <ResponsiveContainer width="100%" height="85%">
              <BarChart data={attendanceData}>
                <CartesianGrid strokeDasharray="3 3" />
                <XAxis dataKey="name" />
                <YAxis />
                <Tooltip />
                <Bar dataKey="present" fill="#4caf50" name="Present" />
                <Bar dataKey="absent" fill="#f44336" name="Absent" />
              </BarChart>
            </ResponsiveContainer>
          </Paper>
        </Grid>

        {/* Department Distribution */}
        <Grid
          size={{
            xs: 12,
            md: 6
          }}>
          <Paper sx={{ p: 3, height: '400px' }}>
            <Typography variant="h6" gutterBottom>
              Department Distribution
            </Typography>
            <ResponsiveContainer width="100%" height="85%">
              <PieChart>
                <Pie
                  data={departmentData}
                  cx="50%"
                  cy="50%"
                  labelLine={false}
                  label={({ name, percent }) => `${name} ${((percent || 0) * 100).toFixed(0)}%`}
                  outerRadius={80}
                  fill="#8884d8"
                  dataKey="value"
                >
                  {departmentData.map((entry, index) => (
                    <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                  ))}
                </Pie>
                <Tooltip />
              </PieChart>
            </ResponsiveContainer>
          </Paper>
        </Grid>

        {/* Today's Attendance */}
        <Grid
          size={{
            xs: 12,
            md: 6
          }}>
          <Paper sx={{ p: 3 }}>
            <Typography variant="h6" gutterBottom>
              Today's Attendance
            </Typography>
            <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
              <Schedule sx={{ mr: 1, color: 'primary.main' }} />
              <Typography variant="h4">
                {stats.todayAttendance}/{stats.totalEmployees}
              </Typography>
            </Box>
            <LinearProgress
              variant="determinate"
              value={attendanceRate}
              sx={{ mb: 1, height: 8, borderRadius: 4 }}
            />
            <Typography variant="body2" color="textSecondary">
              {attendanceRate}% attendance rate
            </Typography>
          </Paper>
        </Grid>

        {/* Payroll Trend */}
        <Grid
          size={{
            xs: 12,
            md: 6
          }}>
          <Paper sx={{ p: 3, height: '300px' }}>
            <Typography variant="h6" gutterBottom>
              Monthly Payroll Trend
            </Typography>
            <ResponsiveContainer width="100%" height="85%">
              <LineChart data={payrollTrend}>
                <CartesianGrid strokeDasharray="3 3" />
                <XAxis dataKey="month" />
                <YAxis />
                <Tooltip formatter={(value) => [`$${value.toLocaleString()}`, 'Payroll']} />
                <Line type="monotone" dataKey="amount" stroke="#1976d2" strokeWidth={3} />
              </LineChart>
            </ResponsiveContainer>
          </Paper>
        </Grid>
      </Grid>
    </Box>
  );
};