import React, { useState, useEffect } from 'react';
import {
  Box,
  Paper,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TablePagination,
  TableRow,
  Typography,
  Button,
  TextField,
  InputAdornment,
  Chip,
  Avatar,
  IconButton,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Grid,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Fab,
  Tooltip,
} from '@mui/material';
import {
  Search,
  Add,
  Edit,
  Delete,
  Visibility,
  FilterList,
} from '@mui/icons-material';
import { Employee } from '../types';
import { employeeService } from '../services/employeeService';
import { useFormValidation, VALIDATION_SCHEMAS } from '../utils/validation';

interface EmployeeDialogProps {
  open: boolean;
  employee: Employee | null;
  onClose: () => void;
  onSave: (employee: Partial<Employee>) => void;
  mode: 'view' | 'edit' | 'create';
}

const EmployeeDialog: React.FC<EmployeeDialogProps> = ({
  open,
  employee,
  onClose,
  onSave,
  mode,
}) => {
  const [formData, setFormData] = useState<Partial<Employee>>({
    firstName: '',
    lastName: '',
    email: '',
    phone: '',
    department: '',
    position: '',
    salary: 0,
    hireDate: '',
    status: 'active',
    address: '',
    dateOfBirth: '',
    emergencyContact: {
      name: '',
      phone: '',
      relationship: '',
    },
  });

  const {
    errors,
    validateForm,
    handleFieldChange,
    clearErrors,
    hasError,
    getErrorMessage,
  } = useFormValidation(VALIDATION_SCHEMAS.EMPLOYEE);

  useEffect(() => {
    if (employee) {
      setFormData(employee);
    } else if (mode === 'create') {
      setFormData({
        firstName: '',
        lastName: '',
        email: '',
        phone: '',
        department: '',
        position: '',
        salary: 0,
        hireDate: new Date().toISOString().split('T')[0],
        status: 'active',
        address: '',
        dateOfBirth: '',
        emergencyContact: {
          name: '',
          phone: '',
          relationship: '',
        },
      });
    }
  }, [employee, mode]);

  const handleFieldChangeWithValidation = (fieldName: string, value: any) => {
    setFormData(prev => ({ ...prev, [fieldName]: value }));
    handleFieldChange(fieldName, value);
  };

  const handleSubmit = () => {
    const validation = validateForm(formData);
    if (!validation.isValid) {
      return; // Don't submit if validation fails
    }
    onSave(formData);
    clearErrors();
  };

  const isReadonly = mode === 'view';

  return (
    <Dialog open={open} onClose={onClose} maxWidth="md" fullWidth>
      <DialogTitle>
        {mode === 'create' ? 'Add New Employee' : 
         mode === 'edit' ? 'Edit Employee' : 'Employee Details'}
      </DialogTitle>
      <DialogContent>
        <Grid container spacing={2} sx={{ mt: 1 }}>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="First Name"
              value={formData.firstName || ''}
              onChange={(e) => handleFieldChangeWithValidation('firstName', e.target.value)}
              disabled={isReadonly}
              required
              error={hasError('firstName')}
              helperText={getErrorMessage('firstName')}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Last Name"
              value={formData.lastName || ''}
              onChange={(e) => handleFieldChangeWithValidation('lastName', e.target.value)}
              disabled={isReadonly}
              required
              error={hasError('lastName')}
              helperText={getErrorMessage('lastName')}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Email"
              type="email"
              value={formData.email || ''}
              onChange={(e) => handleFieldChangeWithValidation('email', e.target.value)}
              disabled={isReadonly}
              required
              error={hasError('email')}
              helperText={getErrorMessage('email')}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Phone"
              value={formData.phone || ''}
              onChange={(e) => handleFieldChangeWithValidation('phone', e.target.value)}
              disabled={isReadonly}
              error={hasError('phone')}
              helperText={getErrorMessage('phone')}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Department"
              value={formData.department || ''}
              onChange={(e) => handleFieldChangeWithValidation('department', e.target.value)}
              disabled={isReadonly}
              required
              error={hasError('department')}
              helperText={getErrorMessage('department')}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Position"
              value={formData.position || ''}
              onChange={(e) => handleFieldChangeWithValidation('position', e.target.value)}
              disabled={isReadonly}
              required
              error={hasError('position')}
              helperText={getErrorMessage('position')}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Salary"
              type="number"
              value={formData.salary || ''}
              onChange={(e) => handleFieldChangeWithValidation('salary', Number(e.target.value))}
              disabled={isReadonly}
              error={hasError('salary')}
              helperText={getErrorMessage('salary')}
              InputProps={{
                startAdornment: <InputAdornment position="start">$</InputAdornment>,
              }}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Hire Date"
              type="date"
              value={formData.hireDate || ''}
              onChange={(e) => handleFieldChangeWithValidation('hireDate', e.target.value)}
              disabled={isReadonly}
              required
              error={hasError('hireDate')}
              helperText={getErrorMessage('hireDate')}
              InputLabelProps={{ shrink: true }}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <FormControl fullWidth>
              <InputLabel>Status</InputLabel>
              <Select
                value={formData.status || 'active'}
                onChange={(e) => setFormData({ ...formData, status: e.target.value as any })}
                disabled={isReadonly}
              >
                <MenuItem value="active">Active</MenuItem>
                <MenuItem value="inactive">Inactive</MenuItem>
                <MenuItem value="terminated">Terminated</MenuItem>
              </Select>
            </FormControl>
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Date of Birth"
              type="date"
              value={formData.dateOfBirth || ''}
              onChange={(e) => setFormData({ ...formData, dateOfBirth: e.target.value })}
              disabled={isReadonly}
              InputLabelProps={{ shrink: true }}
            />
          </Grid>
          <Grid size={12}>
            <TextField
              fullWidth
              label="Address"
              multiline
              rows={2}
              value={formData.address || ''}
              onChange={(e) => setFormData({ ...formData, address: e.target.value })}
              disabled={isReadonly}
            />
          </Grid>
          <Grid size={12}>
            <Typography variant="h6" gutterBottom sx={{ mt: 2 }}>
              Emergency Contact
            </Typography>
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 4
            }}>
            <TextField
              fullWidth
              label="Contact Name"
              value={formData.emergencyContact?.name || ''}
              onChange={(e) => setFormData({ 
                ...formData, 
                emergencyContact: { ...formData.emergencyContact, name: e.target.value } 
              })}
              disabled={isReadonly}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 4
            }}>
            <TextField
              fullWidth
              label="Contact Phone"
              value={formData.emergencyContact?.phone || ''}
              onChange={(e) => setFormData({ 
                ...formData, 
                emergencyContact: { ...formData.emergencyContact, phone: e.target.value } 
              })}
              disabled={isReadonly}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 4
            }}>
            <TextField
              fullWidth
              label="Relationship"
              value={formData.emergencyContact?.relationship || ''}
              onChange={(e) => setFormData({ 
                ...formData, 
                emergencyContact: { ...formData.emergencyContact, relationship: e.target.value } 
              })}
              disabled={isReadonly}
            />
          </Grid>
        </Grid>
      </DialogContent>
      <DialogActions>
        <Button onClick={onClose}>Cancel</Button>
        {!isReadonly && (
          <Button onClick={handleSubmit} variant="contained">
            {mode === 'create' ? 'Add Employee' : 'Save Changes'}
          </Button>
        )}
      </DialogActions>
    </Dialog>
  );
};

export const Employees: React.FC = () => {
  const [employees, setEmployees] = useState<Employee[]>([]);
  const [filteredEmployees, setFilteredEmployees] = useState<Employee[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [page, setPage] = useState(0);
  const [rowsPerPage, setRowsPerPage] = useState(10);
  const [selectedEmployee, setSelectedEmployee] = useState<Employee | null>(null);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [dialogMode, setDialogMode] = useState<'view' | 'edit' | 'create'>('view');
  const [departmentFilter, setDepartmentFilter] = useState('');

  useEffect(() => {
    loadEmployees();
  }, []);

  useEffect(() => {
    filterEmployees();
  }, [employees, searchTerm, departmentFilter]);

  const loadEmployees = async () => {
    try {
      setLoading(true);
      const data = await employeeService.getAllEmployees();
      setEmployees(data);
    } catch (error) {
      console.error('Failed to load employees:', error);
    } finally {
      setLoading(false);
    }
  };

  const filterEmployees = () => {
    let filtered = employees;

    if (searchTerm) {
      filtered = filtered.filter(emp =>
        emp.firstName.toLowerCase().includes(searchTerm.toLowerCase()) ||
        emp.lastName.toLowerCase().includes(searchTerm.toLowerCase()) ||
        emp.email.toLowerCase().includes(searchTerm.toLowerCase()) ||
        emp.employeeId.toLowerCase().includes(searchTerm.toLowerCase()) ||
        emp.department.toLowerCase().includes(searchTerm.toLowerCase()) ||
        emp.position.toLowerCase().includes(searchTerm.toLowerCase())
      );
    }

    if (departmentFilter) {
      filtered = filtered.filter(emp => emp.department === departmentFilter);
    }

    setFilteredEmployees(filtered);
    setPage(0);
  };

  const handleView = (employee: Employee) => {
    setSelectedEmployee(employee);
    setDialogMode('view');
    setDialogOpen(true);
  };

  const handleEdit = (employee: Employee) => {
    setSelectedEmployee(employee);
    setDialogMode('edit');
    setDialogOpen(true);
  };

  const handleCreate = () => {
    setSelectedEmployee(null);
    setDialogMode('create');
    setDialogOpen(true);
  };

  const handleDelete = async (employee: Employee) => {
    if (window.confirm(`Are you sure you want to delete ${employee.firstName} ${employee.lastName}?`)) {
      try {
        await employeeService.deleteEmployee(employee.id);
        loadEmployees();
      } catch (error) {
        console.error('Failed to delete employee:', error);
      }
    }
  };

  const handleSave = async (employeeData: Partial<Employee>) => {
    try {
      if (dialogMode === 'create') {
        await employeeService.createEmployee(employeeData as Omit<Employee, 'id'>);
      } else if (dialogMode === 'edit' && selectedEmployee) {
        await employeeService.updateEmployee(selectedEmployee.id, employeeData);
      }
      setDialogOpen(false);
      loadEmployees();
    } catch (error) {
      console.error('Failed to save employee:', error);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active': return 'success';
      case 'inactive': return 'warning';
      case 'terminated': return 'error';
      default: return 'default';
    }
  };

  const departments = [...new Set(employees.map(emp => emp.department))];

  return (
    <Box>
      <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
        <Typography variant="h4" gutterBottom>
          Employee Management
        </Typography>
        <Tooltip title="Add Employee">
          <Fab color="primary" aria-label="add" onClick={handleCreate}>
            <Add />
          </Fab>
        </Tooltip>
      </Box>

      <Paper sx={{ mb: 2, p: 2 }}>
        <Box sx={{ display: 'flex', gap: 2, alignItems: 'center', flexWrap: 'wrap' }}>
          <TextField
            size="small"
            placeholder="Search employees..."
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
            InputProps={{
              startAdornment: (
                <InputAdornment position="start">
                  <Search />
                </InputAdornment>
              ),
            }}
            sx={{ minWidth: 250 }}
          />
          <FormControl size="small" sx={{ minWidth: 150 }}>
            <InputLabel>Department</InputLabel>
            <Select
              value={departmentFilter}
              onChange={(e) => setDepartmentFilter(e.target.value)}
              label="Department"
            >
              <MenuItem value="">All Departments</MenuItem>
              {departments.map(dept => (
                <MenuItem key={dept} value={dept}>{dept}</MenuItem>
              ))}
            </Select>
          </FormControl>
          <Button
            startIcon={<FilterList />}
            onClick={() => {
              setSearchTerm('');
              setDepartmentFilter('');
            }}
          >
            Clear Filters
          </Button>
        </Box>
      </Paper>

      <TableContainer component={Paper}>
        <Table>
          <TableHead>
            <TableRow>
              <TableCell>Employee</TableCell>
              <TableCell>ID</TableCell>
              <TableCell>Department</TableCell>
              <TableCell>Position</TableCell>
              <TableCell>Status</TableCell>
              <TableCell>Hire Date</TableCell>
              <TableCell>Actions</TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {filteredEmployees
              .slice(page * rowsPerPage, page * rowsPerPage + rowsPerPage)
              .map((employee) => (
                <TableRow key={employee.id} hover>
                  <TableCell>
                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                      <Avatar>
                        {employee.firstName[0]}{employee.lastName[0]}
                      </Avatar>
                      <Box>
                        <Typography variant="body2" fontWeight="bold">
                          {employee.firstName} {employee.lastName}
                        </Typography>
                        <Typography variant="body2" color="textSecondary">
                          {employee.email}
                        </Typography>
                      </Box>
                    </Box>
                  </TableCell>
                  <TableCell>{employee.employeeId}</TableCell>
                  <TableCell>{employee.department}</TableCell>
                  <TableCell>{employee.position}</TableCell>
                  <TableCell>
                    <Chip
                      label={employee.status}
                      color={getStatusColor(employee.status) as any}
                      size="small"
                    />
                  </TableCell>
                  <TableCell>
                    {new Date(employee.hireDate).toLocaleDateString()}
                  </TableCell>
                  <TableCell>
                    <IconButton onClick={() => handleView(employee)} size="small">
                      <Visibility />
                    </IconButton>
                    <IconButton onClick={() => handleEdit(employee)} size="small">
                      <Edit />
                    </IconButton>
                    <IconButton onClick={() => handleDelete(employee)} size="small" color="error">
                      <Delete />
                    </IconButton>
                  </TableCell>
                </TableRow>
              ))}
          </TableBody>
        </Table>
        <TablePagination
          rowsPerPageOptions={[5, 10, 25]}
          component="div"
          count={filteredEmployees.length}
          rowsPerPage={rowsPerPage}
          page={page}
          onPageChange={(event, newPage) => setPage(newPage)}
          onRowsPerPageChange={(event) => {
            setRowsPerPage(parseInt(event.target.value, 10));
            setPage(0);
          }}
        />
      </TableContainer>

      <EmployeeDialog
        open={dialogOpen}
        employee={selectedEmployee}
        onClose={() => setDialogOpen(false)}
        onSave={handleSave}
        mode={dialogMode}
      />
    </Box>
  );
};