import React, { useState, useEffect } from 'react';
import {
  Box,
  Typography,
  LinearProgress,
  useTheme,
  useMediaQuery,
  Grid,
  Tabs,
  Tab,
} from '@mui/material';
import {
  People,
  Business,
  Schedule,
  Payment,
  TrendingUp,
  EventNote,
} from '@mui/icons-material';
import { DashboardStats } from '../types';
import { MobileStatsGrid } from '../components/Common/MobileTable';
import {
  ResponsiveBarChart,
  ResponsivePieChart,
  ResponsiveLineChart,
} from '../components/Common/MobileCharts';

const mockStats: DashboardStats = {
  totalEmployees: 145,
  activeEmployees: 138,
  totalDepartments: 8,
  pendingLeaves: 12,
  todayAttendance: 92,
  monthlyPayroll: 485000,
};

const attendanceData = [
  { name: 'Mon', present: 135, absent: 10 },
  { name: 'Tue', present: 132, absent: 13 },
  { name: 'Wed', present: 140, absent: 5 },
  { name: 'Thu', present: 138, absent: 7 },
  { name: 'Fri', present: 130, absent: 15 },
];

const departmentData = [
  { name: 'Engineering', value: 45 },
  { name: 'Sales', value: 32 },
  { name: 'Marketing', value: 28 },
  { name: 'HR', value: 15 },
  { name: 'Finance', value: 25 },
];

const payrollTrend = [
  { month: 'Jan', amount: 450000 },
  { month: 'Feb', amount: 465000 },
  { month: 'Mar', amount: 470000 },
  { month: 'Apr', amount: 485000 },
  { month: 'May', amount: 480000 },
  { month: 'Jun', amount: 485000 },
];

interface TabPanelProps {
  children?: React.ReactNode;
  index: number;
  value: number;
}

const TabPanel: React.FC<TabPanelProps> = ({ children, value, index }) => (
  <div hidden={value !== index}>
    {value === index && <Box sx={{ py: 2 }}>{children}</Box>}
  </div>
);

export const MobileDashboard: React.FC = () => {
  const [stats, setStats] = useState<DashboardStats>(mockStats);
  const [loading, setLoading] = useState(true);
  const [tabValue, setTabValue] = useState(0);
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down('md'));

  useEffect(() => {
    setTimeout(() => {
      setStats(mockStats);
      setLoading(false);
    }, 1000);
  }, []);

  const attendanceRate = Math.round((stats.todayAttendance / stats.totalEmployees) * 100);

  if (loading) {
    return (
      <Box sx={{ width: '100%', mt: 2 }}>
        <LinearProgress />
      </Box>
    );
  }

  const statsData = [
    {
      title: 'Total Employees',
      value: stats.totalEmployees,
      icon: <People sx={{ fontSize: { xs: 20, sm: 24 } }} />,
      color: '#1976d2',
      change: '+5% from last month'
    },
    {
      title: 'Active Employees',
      value: stats.activeEmployees,
      icon: <TrendingUp sx={{ fontSize: { xs: 20, sm: 24 } }} />,
      color: '#2e7d32'
    },
    {
      title: 'Departments',
      value: stats.totalDepartments,
      icon: <Business sx={{ fontSize: { xs: 20, sm: 24 } }} />,
      color: '#ed6c02'
    },
    {
      title: 'Pending Leaves',
      value: stats.pendingLeaves,
      icon: <EventNote sx={{ fontSize: { xs: 20, sm: 24 } }} />,
      color: '#d32f2f'
    },
    {
      title: 'Today Attendance',
      value: `${attendanceRate}%`,
      icon: <Schedule sx={{ fontSize: { xs: 20, sm: 24 } }} />,
      color: '#1976d2'
    },
    {
      title: 'Monthly Payroll',
      value: `$${(stats.monthlyPayroll / 1000).toFixed(0)}K`,
      icon: <Payment sx={{ fontSize: { xs: 20, sm: 24 } }} />,
      color: '#7b1fa2'
    }
  ];

  return (
    <Box>
      <Typography 
        variant={isMobile ? "h5" : "h4"} 
        gutterBottom
        sx={{ 
          mb: { xs: 2, sm: 3 },
          fontWeight: 600
        }}
      >
        Dashboard Overview
      </Typography>
      {/* Stats Grid */}
      <MobileStatsGrid 
        stats={statsData} 
        columns={{ xs: 2, sm: 3, md: 4 }}
      />
      {/* Charts Navigation for Mobile */}
      {isMobile ? (
        <>
          <Box sx={{ borderBottom: 1, borderColor: 'divider', mb: 2 }}>
            <Tabs 
              value={tabValue} 
              onChange={(e, newValue) => setTabValue(newValue)}
              variant="scrollable"
              scrollButtons="auto"
              allowScrollButtonsMobile
            >
              <Tab label="Attendance" />
              <Tab label="Departments" />
              <Tab label="Payroll" />
            </Tabs>
          </Box>

          <TabPanel value={tabValue} index={0}>
            <ResponsiveBarChart
              data={attendanceData}
              dataKey="present"
              xAxisKey="name"
              title="Weekly Attendance"
              color="#4caf50"
            />
          </TabPanel>

          <TabPanel value={tabValue} index={1}>
            <ResponsivePieChart
              data={departmentData}
              dataKey="value"
              nameKey="name"
              title="Department Distribution"
            />
          </TabPanel>

          <TabPanel value={tabValue} index={2}>
            <ResponsiveLineChart
              data={payrollTrend}
              dataKey="amount"
              xAxisKey="month"
              title="Payroll Trend"
              color="#1976d2"
            />
          </TabPanel>
        </>
      ) : (
        <Grid container spacing={3}>
          {/* Desktop Layout - Grid of Charts */}
          <Grid
            size={{
              xs: 12,
              md: 6
            }}>
            <ResponsiveBarChart
              data={attendanceData}
              dataKey="present"
              xAxisKey="name"
              title="Weekly Attendance"
              color="#4caf50"
            />
          </Grid>

          <Grid
            size={{
              xs: 12,
              md: 6
            }}>
            <ResponsivePieChart
              data={departmentData}
              dataKey="value"
              nameKey="name"
              title="Department Distribution"
            />
          </Grid>

          <Grid size={12}>
            <ResponsiveLineChart
              data={payrollTrend}
              dataKey="amount"
              xAxisKey="month"
              title="Monthly Payroll Trend"
              color="#1976d2"
            />
          </Grid>
        </Grid>
      )}
    </Box>
  );
};