import React, { useState, useEffect } from 'react';
import {
  Box,
  Paper,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TablePagination,
  TableRow,
  Typography,
  Button,
  TextField,
  InputAdornment,
  Chip,
  Card,
  CardContent,
  Grid,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Avatar,
  IconButton,
  Tabs,
  Tab,
  Divider,
  List,
  ListItem,
  ListItemText,
} from '@mui/material';
import {
  Search,
  Add,
  Visibility,
  Edit,
  Delete,
  Payment,
  TrendingUp,
  AccountBalance,
  Receipt,
  FilterList,
  GetApp,
} from '@mui/icons-material';
import { PayrollRecord } from '../types';
import { format } from 'date-fns';
import { useFormValidation, VALIDATION_SCHEMAS } from '../utils/validation';

const mockPayrollRecords: PayrollRecord[] = [
  {
    id: '1',
    employeeId: '1',
    month: 'June',
    year: 2023,
    basicSalary: 8000,
    allowances: 1500,
    deductions: 800,
    overtime: 500,
    netSalary: 9200,
    payDate: '2023-06-30',
    status: 'paid',
  },
  {
    id: '2',
    employeeId: '2',
    month: 'June',
    year: 2023,
    basicSalary: 6500,
    allowances: 1000,
    deductions: 650,
    overtime: 300,
    netSalary: 7150,
    payDate: '2023-06-30',
    status: 'paid',
  },
  {
    id: '3',
    employeeId: '3',
    month: 'June',
    year: 2023,
    basicSalary: 5500,
    allowances: 800,
    deductions: 550,
    overtime: 200,
    netSalary: 5950,
    payDate: '2023-06-30',
    status: 'processed',
  },
  {
    id: '4',
    employeeId: '4',
    month: 'June',
    year: 2023,
    basicSalary: 5000,
    allowances: 750,
    deductions: 500,
    overtime: 150,
    netSalary: 5400,
    payDate: '2023-06-30',
    status: 'processed',
  },
  {
    id: '5',
    employeeId: '5',
    month: 'June',
    year: 2023,
    basicSalary: 6000,
    allowances: 900,
    deductions: 600,
    overtime: 250,
    netSalary: 6550,
    payDate: '',
    status: 'draft',
  },
];

const mockEmployeeNames: { [key: string]: string } = {
  '1': 'John Doe',
  '2': 'Sarah Johnson',
  '3': 'Michael Brown',
  '4': 'Emily Davis',
  '5': 'David Wilson',
};

interface PayrollDialogProps {
  open: boolean;
  payroll: PayrollRecord | null;
  onClose: () => void;
  onSave?: (payroll: Partial<PayrollRecord>) => void;
  onProcess?: (payrollId: string) => void;
  onPay?: (payrollId: string) => void;
  mode: 'view' | 'edit' | 'create';
}

const PayrollDialog: React.FC<PayrollDialogProps> = ({
  open,
  payroll,
  onClose,
  onSave,
  onProcess,
  onPay,
  mode,
}) => {
  const [formData, setFormData] = useState<Partial<PayrollRecord>>({
    employeeId: '',
    month: format(new Date(), 'MMMM'),
    year: new Date().getFullYear(),
    basicSalary: 0,
    allowances: 0,
    deductions: 0,
    overtime: 0,
    netSalary: 0,
    status: 'draft',
  });

  const {
    errors,
    validateForm,
    handleFieldChange,
    clearErrors,
    hasError,
    getErrorMessage,
  } = useFormValidation(VALIDATION_SCHEMAS.PAYROLL);

  useEffect(() => {
    if (payroll) {
      setFormData(payroll);
    } else if (mode === 'create') {
      setFormData({
        employeeId: '',
        month: format(new Date(), 'MMMM'),
        year: new Date().getFullYear(),
        basicSalary: 0,
        allowances: 0,
        deductions: 0,
        overtime: 0,
        netSalary: 0,
        status: 'draft',
      });
    }
  }, [payroll, mode]);

  useEffect(() => {
    const netSalary = (formData.basicSalary || 0) + (formData.allowances || 0) + (formData.overtime || 0) - (formData.deductions || 0);
    setFormData(prev => ({ ...prev, netSalary }));
  }, [formData.basicSalary, formData.allowances, formData.deductions, formData.overtime]);

  const handleFieldChangeWithValidation = (fieldName: string, value: any) => {
    setFormData(prev => ({ ...prev, [fieldName]: value }));
    handleFieldChange(fieldName, value);
  };

  const handleSubmit = () => {
    const validation = validateForm(formData);
    if (!validation.isValid) {
      return;
    }
    if (onSave) {
      onSave(formData);
      clearErrors();
    }
  };

  const handleProcess = () => {
    if (onProcess && payroll) {
      onProcess(payroll.id);
    }
  };

  const handlePay = () => {
    if (onPay && payroll) {
      onPay(payroll.id);
    }
  };

  const isReadonly = mode === 'view';
  const canProcess = payroll?.status === 'draft';
  const canPay = payroll?.status === 'processed';

  return (
    <Dialog open={open} onClose={onClose} maxWidth="md" fullWidth>
      <DialogTitle>
        {mode === 'create' ? 'Create Payroll Record' : 
         mode === 'edit' ? 'Edit Payroll Record' : 'Payroll Details'}
      </DialogTitle>
      <DialogContent>
        <Grid container spacing={2} sx={{ mt: 1 }}>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Employee ID"
              value={formData.employeeId || ''}
              onChange={(e) => handleFieldChangeWithValidation('employeeId', e.target.value)}
              disabled={isReadonly}
              required
              error={hasError('employeeId')}
              helperText={getErrorMessage('employeeId')}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Employee Name"
              value={mockEmployeeNames[formData.employeeId || ''] || ''}
              disabled
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 4
            }}>
            <TextField
              fullWidth
              label="Month"
              value={formData.month || ''}
              onChange={(e) => handleFieldChangeWithValidation('month', e.target.value)}
              disabled={isReadonly}
              required
              error={hasError('month')}
              helperText={getErrorMessage('month')}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 4
            }}>
            <TextField
              fullWidth
              label="Year"
              type="number"
              value={formData.year || ''}
              onChange={(e) => handleFieldChangeWithValidation('year', Number(e.target.value))}
              disabled={isReadonly}
              required
              error={hasError('year')}
              helperText={getErrorMessage('year')}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 4
            }}>
            <TextField
              fullWidth
              label="Status"
              value={formData.status}
              disabled
            />
          </Grid>
          
          <Grid size={12}>
            <Divider sx={{ my: 2 }}>
              <Typography variant="h6">Salary Breakdown</Typography>
            </Divider>
          </Grid>
          
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Basic Salary"
              type="number"
              value={formData.basicSalary || ''}
              onChange={(e) => handleFieldChangeWithValidation('basicSalary', Number(e.target.value))}
              disabled={isReadonly}
              required
              error={hasError('basicSalary')}
              helperText={getErrorMessage('basicSalary')}
              InputProps={{
                startAdornment: <InputAdornment position="start">$</InputAdornment>,
              }}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Allowances"
              type="number"
              value={formData.allowances || ''}
              onChange={(e) => handleFieldChangeWithValidation('allowances', Number(e.target.value))}
              disabled={isReadonly}
              error={hasError('allowances')}
              helperText={getErrorMessage('allowances')}
              InputProps={{
                startAdornment: <InputAdornment position="start">$</InputAdornment>,
              }}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Overtime"
              type="number"
              value={formData.overtime || ''}
              onChange={(e) => handleFieldChangeWithValidation('overtime', Number(e.target.value))}
              disabled={isReadonly}
              error={hasError('overtime')}
              helperText={getErrorMessage('overtime')}
              InputProps={{
                startAdornment: <InputAdornment position="start">$</InputAdornment>,
              }}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Deductions"
              type="number"
              value={formData.deductions || ''}
              onChange={(e) => handleFieldChangeWithValidation('deductions', Number(e.target.value))}
              disabled={isReadonly}
              error={hasError('deductions')}
              helperText={getErrorMessage('deductions')}
              InputProps={{
                startAdornment: <InputAdornment position="start">$</InputAdornment>,
              }}
            />
          </Grid>
          <Grid size={12}>
            <Card variant="outlined" sx={{ bgcolor: 'primary.light', color: 'primary.contrastText' }}>
              <CardContent>
                <Typography variant="h5" align="center">
                  Net Salary: ${formData.netSalary?.toLocaleString()}
                </Typography>
              </CardContent>
            </Card>
          </Grid>
          {formData.payDate && (
            <Grid size={12}>
              <TextField
                fullWidth
                label="Pay Date"
                value={formData.payDate}
                disabled
              />
            </Grid>
          )}
        </Grid>

        {mode === 'view' && payroll && (
          <Box sx={{ mt: 3 }}>
            <Divider sx={{ mb: 2 }}>
              <Typography variant="h6">Salary Breakdown</Typography>
            </Divider>
            <List>
              <ListItem>
                <ListItemText 
                  primary="Basic Salary" 
                  secondary={`$${payroll.basicSalary.toLocaleString()}`}
                />
              </ListItem>
              <ListItem>
                <ListItemText 
                  primary="Allowances" 
                  secondary={`$${payroll.allowances.toLocaleString()}`}
                />
              </ListItem>
              <ListItem>
                <ListItemText 
                  primary="Overtime" 
                  secondary={`$${payroll.overtime.toLocaleString()}`}
                />
              </ListItem>
              <ListItem sx={{ color: 'error.main' }}>
                <ListItemText 
                  primary="Deductions" 
                  secondary={`-$${payroll.deductions.toLocaleString()}`}
                />
              </ListItem>
              <Divider />
              <ListItem sx={{ bgcolor: 'success.light' }}>
                <ListItemText 
                  primary={<Typography variant="h6">Net Salary</Typography>}
                  secondary={<Typography variant="h5">${payroll.netSalary.toLocaleString()}</Typography>}
                />
              </ListItem>
            </List>
          </Box>
        )}
      </DialogContent>
      <DialogActions>
        <Button onClick={onClose}>Close</Button>
        {canProcess && (
          <Button onClick={handleProcess} variant="outlined" color="primary">
            Process Payroll
          </Button>
        )}
        {canPay && (
          <Button onClick={handlePay} variant="contained" color="success">
            Mark as Paid
          </Button>
        )}
        {!isReadonly && (
          <Button onClick={handleSubmit} variant="contained">
            {mode === 'create' ? 'Create Record' : 'Save Changes'}
          </Button>
        )}
      </DialogActions>
    </Dialog>
  );
};

interface TabPanelProps {
  children?: React.ReactNode;
  index: number;
  value: number;
}

const TabPanel: React.FC<TabPanelProps> = ({ children, value, index }) => (
  <div hidden={value !== index}>
    {value === index && <Box sx={{ py: 3 }}>{children}</Box>}
  </div>
);

export const Payroll: React.FC = () => {
  const [payrolls, setPayrolls] = useState<PayrollRecord[]>(mockPayrollRecords);
  const [filteredPayrolls, setFilteredPayrolls] = useState<PayrollRecord[]>(mockPayrollRecords);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState('');
  const [monthFilter, setMonthFilter] = useState('');
  const [page, setPage] = useState(0);
  const [rowsPerPage, setRowsPerPage] = useState(10);
  const [selectedPayroll, setSelectedPayroll] = useState<PayrollRecord | null>(null);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [dialogMode, setDialogMode] = useState<'view' | 'edit' | 'create'>('view');
  const [tabValue, setTabValue] = useState(0);

  useEffect(() => {
    filterPayrolls();
  }, [payrolls, searchTerm, statusFilter, monthFilter]);

  const filterPayrolls = () => {
    let filtered = payrolls;

    if (searchTerm) {
      filtered = filtered.filter(payroll =>
        mockEmployeeNames[payroll.employeeId]?.toLowerCase().includes(searchTerm.toLowerCase()) ||
        payroll.employeeId.toLowerCase().includes(searchTerm.toLowerCase())
      );
    }

    if (statusFilter) {
      filtered = filtered.filter(payroll => payroll.status === statusFilter);
    }

    if (monthFilter) {
      filtered = filtered.filter(payroll => payroll.month === monthFilter);
    }

    setFilteredPayrolls(filtered);
    setPage(0);
  };

  const handleView = (payroll: PayrollRecord) => {
    setSelectedPayroll(payroll);
    setDialogMode('view');
    setDialogOpen(true);
  };

  const handleEdit = (payroll: PayrollRecord) => {
    setSelectedPayroll(payroll);
    setDialogMode('edit');
    setDialogOpen(true);
  };

  const handleCreate = () => {
    setSelectedPayroll(null);
    setDialogMode('create');
    setDialogOpen(true);
  };

  const handleDelete = (payroll: PayrollRecord) => {
    if (window.confirm(`Are you sure you want to delete this payroll record?`)) {
      setPayrolls(prev => prev.filter(p => p.id !== payroll.id));
    }
  };

  const handleSave = (payrollData: Partial<PayrollRecord>) => {
    if (dialogMode === 'create') {
      const newPayroll: PayrollRecord = {
        ...payrollData,
        id: Date.now().toString(),
      } as PayrollRecord;
      setPayrolls(prev => [...prev, newPayroll]);
    } else if (dialogMode === 'edit' && selectedPayroll) {
      setPayrolls(prev =>
        prev.map(p => p.id === selectedPayroll.id ? { ...p, ...payrollData } : p)
      );
    }
    setDialogOpen(false);
  };

  const handleProcess = (payrollId: string) => {
    setPayrolls(prev =>
      prev.map(p => p.id === payrollId ? { ...p, status: 'processed' } : p)
    );
    setDialogOpen(false);
  };

  const handlePay = (payrollId: string) => {
    setPayrolls(prev =>
      prev.map(p => p.id === payrollId ? { 
        ...p, 
        status: 'paid', 
        payDate: format(new Date(), 'yyyy-MM-dd') 
      } : p)
    );
    setDialogOpen(false);
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'paid': return 'success';
      case 'processed': return 'warning';
      case 'draft': return 'info';
      default: return 'default';
    }
  };

  const payrollStats = {
    total: payrolls.length,
    draft: payrolls.filter(p => p.status === 'draft').length,
    processed: payrolls.filter(p => p.status === 'processed').length,
    paid: payrolls.filter(p => p.status === 'paid').length,
    totalAmount: payrolls.filter(p => p.status === 'paid').reduce((sum, p) => sum + p.netSalary, 0),
    pendingAmount: payrolls.filter(p => p.status === 'processed').reduce((sum, p) => sum + p.netSalary, 0),
  };

  const months = [...new Set(payrolls.map(p => p.month))];

  return (
    <Box>
      <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
        <Typography variant="h4" gutterBottom>
          Payroll Management
        </Typography>
        <Button variant="contained" startIcon={<Add />} onClick={handleCreate}>
          Create Payroll
        </Button>
      </Box>
      <Box sx={{ borderBottom: 1, borderColor: 'divider', mb: 3 }}>
        <Tabs value={tabValue} onChange={(e, newValue) => setTabValue(newValue)}>
          <Tab label="All Records" />
          <Tab label="Pending Processing" />
          <Tab label="Payroll Summary" />
        </Tabs>
      </Box>
      <TabPanel value={tabValue} index={0}>
        {/* Stats Cards */}
        <Grid container spacing={3} sx={{ mb: 3 }}>
          <Grid
            size={{
              xs: 12,
              sm: 6,
              md: 3
            }}>
            <Card>
              <CardContent sx={{ textAlign: 'center' }}>
                <Receipt sx={{ fontSize: 40, color: 'primary.main', mb: 1 }} />
                <Typography variant="h4">{payrollStats.total}</Typography>
                <Typography color="textSecondary">Total Records</Typography>
              </CardContent>
            </Card>
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6,
              md: 3
            }}>
            <Card>
              <CardContent sx={{ textAlign: 'center' }}>
                <Payment sx={{ fontSize: 40, color: 'success.main', mb: 1 }} />
                <Typography variant="h4">${payrollStats.totalAmount.toLocaleString()}</Typography>
                <Typography color="textSecondary">Total Paid</Typography>
              </CardContent>
            </Card>
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6,
              md: 3
            }}>
            <Card>
              <CardContent sx={{ textAlign: 'center' }}>
                <AccountBalance sx={{ fontSize: 40, color: 'warning.main', mb: 1 }} />
                <Typography variant="h4">${payrollStats.pendingAmount.toLocaleString()}</Typography>
                <Typography color="textSecondary">Pending Payment</Typography>
              </CardContent>
            </Card>
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6,
              md: 3
            }}>
            <Card>
              <CardContent sx={{ textAlign: 'center' }}>
                <TrendingUp sx={{ fontSize: 40, color: 'info.main', mb: 1 }} />
                <Typography variant="h4">{payrollStats.processed}</Typography>
                <Typography color="textSecondary">Ready to Pay</Typography>
              </CardContent>
            </Card>
          </Grid>
        </Grid>

        {/* Filters */}
        <Paper sx={{ mb: 2, p: 2 }}>
          <Box sx={{ display: 'flex', gap: 2, alignItems: 'center', flexWrap: 'wrap' }}>
            <TextField
              size="small"
              placeholder="Search employees..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              InputProps={{
                startAdornment: (
                  <InputAdornment position="start">
                    <Search />
                  </InputAdornment>
                ),
              }}
              sx={{ minWidth: 250 }}
            />
            <FormControl size="small" sx={{ minWidth: 120 }}>
              <InputLabel>Status</InputLabel>
              <Select
                value={statusFilter}
                onChange={(e) => setStatusFilter(e.target.value)}
                label="Status"
              >
                <MenuItem value="">All Status</MenuItem>
                <MenuItem value="draft">Draft</MenuItem>
                <MenuItem value="processed">Processed</MenuItem>
                <MenuItem value="paid">Paid</MenuItem>
              </Select>
            </FormControl>
            <FormControl size="small" sx={{ minWidth: 120 }}>
              <InputLabel>Month</InputLabel>
              <Select
                value={monthFilter}
                onChange={(e) => setMonthFilter(e.target.value)}
                label="Month"
              >
                <MenuItem value="">All Months</MenuItem>
                {months.map(month => (
                  <MenuItem key={month} value={month}>{month}</MenuItem>
                ))}
              </Select>
            </FormControl>
            <Button
              startIcon={<FilterList />}
              onClick={() => {
                setSearchTerm('');
                setStatusFilter('');
                setMonthFilter('');
              }}
            >
              Clear
            </Button>
            <Button
              startIcon={<GetApp />}
              variant="outlined"
            >
              Export
            </Button>
          </Box>
        </Paper>

        {/* Payroll Table */}
        <TableContainer component={Paper}>
          <Table>
            <TableHead>
              <TableRow>
                <TableCell>Employee</TableCell>
                <TableCell>Period</TableCell>
                <TableCell>Basic Salary</TableCell>
                <TableCell>Allowances</TableCell>
                <TableCell>Deductions</TableCell>
                <TableCell>Net Salary</TableCell>
                <TableCell>Status</TableCell>
                <TableCell>Actions</TableCell>
              </TableRow>
            </TableHead>
            <TableBody>
              {filteredPayrolls
                .slice(page * rowsPerPage, page * rowsPerPage + rowsPerPage)
                .map((payroll) => (
                  <TableRow key={payroll.id} hover>
                    <TableCell>
                      <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                        <Avatar sx={{ width: 32, height: 32 }}>
                          {mockEmployeeNames[payroll.employeeId]?.split(' ').map(n => n[0]).join('')}
                        </Avatar>
                        <Box>
                          <Typography variant="body2" fontWeight="bold">
                            {mockEmployeeNames[payroll.employeeId] || `Employee ${payroll.employeeId}`}
                          </Typography>
                          <Typography variant="body2" color="textSecondary">
                            ID: {payroll.employeeId}
                          </Typography>
                        </Box>
                      </Box>
                    </TableCell>
                    <TableCell>
                      <Typography variant="body2">
                        {payroll.month} {payroll.year}
                      </Typography>
                    </TableCell>
                    <TableCell>${payroll.basicSalary.toLocaleString()}</TableCell>
                    <TableCell>${payroll.allowances.toLocaleString()}</TableCell>
                    <TableCell>-${payroll.deductions.toLocaleString()}</TableCell>
                    <TableCell>
                      <Typography variant="body2" fontWeight="bold" color="success.main">
                        ${payroll.netSalary.toLocaleString()}
                      </Typography>
                    </TableCell>
                    <TableCell>
                      <Chip
                        label={payroll.status}
                        color={getStatusColor(payroll.status) as any}
                        size="small"
                      />
                    </TableCell>
                    <TableCell>
                      <IconButton onClick={() => handleView(payroll)} size="small">
                        <Visibility />
                      </IconButton>
                      <IconButton onClick={() => handleEdit(payroll)} size="small">
                        <Edit />
                      </IconButton>
                      <IconButton onClick={() => handleDelete(payroll)} size="small" color="error">
                        <Delete />
                      </IconButton>
                    </TableCell>
                  </TableRow>
                ))}
            </TableBody>
          </Table>
          <TablePagination
            rowsPerPageOptions={[5, 10, 25]}
            component="div"
            count={filteredPayrolls.length}
            rowsPerPage={rowsPerPage}
            page={page}
            onPageChange={(event, newPage) => setPage(newPage)}
            onRowsPerPageChange={(event) => {
              setRowsPerPage(parseInt(event.target.value, 10));
              setPage(0);
            }}
          />
        </TableContainer>
      </TabPanel>
      <TabPanel value={tabValue} index={1}>
        <Typography variant="h6" gutterBottom>
          Records Awaiting Processing ({payrollStats.draft + payrollStats.processed})
        </Typography>
        <TableContainer component={Paper}>
          <Table>
            <TableHead>
              <TableRow>
                <TableCell>Employee</TableCell>
                <TableCell>Period</TableCell>
                <TableCell>Net Salary</TableCell>
                <TableCell>Status</TableCell>
                <TableCell>Actions</TableCell>
              </TableRow>
            </TableHead>
            <TableBody>
              {payrolls.filter(p => p.status === 'draft' || p.status === 'processed').map((payroll) => (
                <TableRow key={payroll.id} hover>
                  <TableCell>
                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                      <Avatar sx={{ width: 32, height: 32 }}>
                        {mockEmployeeNames[payroll.employeeId]?.split(' ').map(n => n[0]).join('')}
                      </Avatar>
                      <Box>
                        <Typography variant="body2" fontWeight="bold">
                          {mockEmployeeNames[payroll.employeeId]}
                        </Typography>
                        <Typography variant="body2" color="textSecondary">
                          ID: {payroll.employeeId}
                        </Typography>
                      </Box>
                    </Box>
                  </TableCell>
                  <TableCell>{payroll.month} {payroll.year}</TableCell>
                  <TableCell>
                    <Typography variant="body2" fontWeight="bold">
                      ${payroll.netSalary.toLocaleString()}
                    </Typography>
                  </TableCell>
                  <TableCell>
                    <Chip
                      label={payroll.status}
                      color={getStatusColor(payroll.status) as any}
                      size="small"
                    />
                  </TableCell>
                  <TableCell>
                    <IconButton onClick={() => handleView(payroll)} size="small">
                      <Visibility />
                    </IconButton>
                  </TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </TableContainer>
      </TabPanel>
      <TabPanel value={tabValue} index={2}>
        <Typography variant="h6" gutterBottom>
          Payroll Summary & Analytics
        </Typography>
        <Typography variant="body1" color="textSecondary">
          Summary reports and analytics features will be implemented here.
        </Typography>
      </TabPanel>
      <PayrollDialog
        open={dialogOpen}
        payroll={selectedPayroll}
        onClose={() => setDialogOpen(false)}
        onSave={handleSave}
        onProcess={handleProcess}
        onPay={handlePay}
        mode={dialogMode}
      />
    </Box>
  );
};