import React, { useState } from 'react';
import {
  Box,
  Paper,
  Typography,
  Card,
  CardContent,
  Grid,
  Button,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  TextField,
  Tabs,
  Tab,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Chip,
} from '@mui/material';
import {
  GetApp,
  PictureAsPdf,
  TableChart,
  Assessment,
  TrendingUp,
  People,
  Schedule,
  Payment,
  EventNote,
} from '@mui/icons-material';
import {
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
  LineChart,
  Line,
  PieChart,
  Pie,
  Cell,
  Area,
  AreaChart,
} from 'recharts';

const COLORS = ['#0088FE', '#00C49F', '#FFBB28', '#FF8042', '#8884D8'];

// Mock data for reports
const employeeGrowthData = [
  { month: 'Jan', employees: 120, hires: 5, terminations: 2 },
  { month: 'Feb', employees: 125, hires: 7, terminations: 2 },
  { month: 'Mar', employees: 130, hires: 8, terminations: 3 },
  { month: 'Apr', employees: 135, hires: 6, terminations: 1 },
  { month: 'May', employees: 140, hires: 7, terminations: 2 },
  { month: 'Jun', employees: 145, hires: 8, terminations: 3 },
];

const attendanceData = [
  { department: 'Engineering', present: 42, absent: 3, late: 2 },
  { department: 'Sales', present: 28, absent: 4, late: 3 },
  { department: 'Marketing', present: 25, absent: 2, late: 1 },
  { department: 'HR', present: 14, absent: 1, late: 0 },
  { department: 'Finance', present: 23, absent: 2, late: 1 },
];

const departmentSalaryData = [
  { name: 'Engineering', value: 2500000, employees: 45 },
  { name: 'Sales', value: 1800000, employees: 32 },
  { name: 'Marketing', value: 1200000, employees: 28 },
  { name: 'HR', value: 800000, employees: 15 },
  { name: 'Finance', value: 1000000, employees: 25 },
];

const leaveAnalytics = [
  { month: 'Jan', sick: 12, vacation: 8, personal: 5, total: 25 },
  { month: 'Feb', sick: 15, vacation: 10, personal: 3, total: 28 },
  { month: 'Mar', sick: 18, vacation: 15, personal: 7, total: 40 },
  { month: 'Apr', sick: 10, vacation: 20, personal: 5, total: 35 },
  { month: 'May', sick: 14, vacation: 25, personal: 8, total: 47 },
  { month: 'Jun', sick: 11, vacation: 18, personal: 4, total: 33 },
];

const payrollTrend = [
  { month: 'Jan', amount: 450000, employees: 120 },
  { month: 'Feb', amount: 465000, employees: 125 },
  { month: 'Mar', amount: 485000, employees: 130 },
  { month: 'Apr', amount: 505000, employees: 135 },
  { month: 'May', amount: 520000, employees: 140 },
  { month: 'Jun', amount: 540000, employees: 145 },
];

const topPerformers = [
  { name: 'John Doe', department: 'Engineering', performance: 98, salary: 95000 },
  { name: 'Sarah Johnson', department: 'Sales', performance: 96, salary: 75000 },
  { name: 'Michael Brown', department: 'Marketing', performance: 94, salary: 62000 },
  { name: 'Emily Davis', department: 'HR', performance: 93, salary: 58000 },
  { name: 'David Wilson', department: 'Finance', performance: 91, salary: 68000 },
];

interface TabPanelProps {
  children?: React.ReactNode;
  index: number;
  value: number;
}

const TabPanel: React.FC<TabPanelProps> = ({ children, value, index }) => (
  <div hidden={value !== index}>
    {value === index && <Box sx={{ py: 3 }}>{children}</Box>}
  </div>
);

export const Reports: React.FC = () => {
  const [tabValue, setTabValue] = useState(0);
  const [reportType, setReportType] = useState('employee');
  const [dateRange, setDateRange] = useState('last6months');

  const exportReport = (format: 'pdf' | 'excel') => {
    // Mock export functionality
    console.log(`Exporting report as ${format}`);
    alert(`Report exported as ${format.toUpperCase()}`);
  };

  return (
    <Box>
      <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
        <Typography variant="h4" gutterBottom>
          Reports & Analytics
        </Typography>
        <Box sx={{ display: 'flex', gap: 1 }}>
          <Button
            variant="outlined"
            startIcon={<PictureAsPdf />}
            onClick={() => exportReport('pdf')}
          >
            Export PDF
          </Button>
          <Button
            variant="contained"
            startIcon={<TableChart />}
            onClick={() => exportReport('excel')}
          >
            Export Excel
          </Button>
        </Box>
      </Box>
      {/* Report Filters */}
      <Paper sx={{ p: 2, mb: 3 }}>
        <Grid container spacing={2} alignItems="center">
          <Grid
            size={{
              xs: 12,
              sm: 4
            }}>
            <FormControl fullWidth size="small">
              <InputLabel>Report Type</InputLabel>
              <Select
                value={reportType}
                onChange={(e) => setReportType(e.target.value)}
                label="Report Type"
              >
                <MenuItem value="employee">Employee Analytics</MenuItem>
                <MenuItem value="attendance">Attendance Report</MenuItem>
                <MenuItem value="payroll">Payroll Analysis</MenuItem>
                <MenuItem value="leave">Leave Management</MenuItem>
                <MenuItem value="performance">Performance Report</MenuItem>
              </Select>
            </FormControl>
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 4
            }}>
            <FormControl fullWidth size="small">
              <InputLabel>Date Range</InputLabel>
              <Select
                value={dateRange}
                onChange={(e) => setDateRange(e.target.value)}
                label="Date Range"
              >
                <MenuItem value="last30days">Last 30 Days</MenuItem>
                <MenuItem value="last3months">Last 3 Months</MenuItem>
                <MenuItem value="last6months">Last 6 Months</MenuItem>
                <MenuItem value="lastyear">Last Year</MenuItem>
                <MenuItem value="custom">Custom Range</MenuItem>
              </Select>
            </FormControl>
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 4
            }}>
            <Button variant="contained" fullWidth startIcon={<Assessment />}>
              Generate Report
            </Button>
          </Grid>
        </Grid>
      </Paper>
      <Box sx={{ borderBottom: 1, borderColor: 'divider', mb: 3 }}>
        <Tabs value={tabValue} onChange={(e, newValue) => setTabValue(newValue)}>
          <Tab icon={<People />} label="Employee Analytics" />
          <Tab icon={<Schedule />} label="Attendance Reports" />
          <Tab icon={<Payment />} label="Payroll Analysis" />
          <Tab icon={<EventNote />} label="Leave Reports" />
          <Tab icon={<TrendingUp />} label="Performance" />
        </Tabs>
      </Box>
      <TabPanel value={tabValue} index={0}>
        {/* Employee Analytics */}
        <Grid container spacing={3}>
          <Grid
            size={{
              xs: 12,
              md: 8
            }}>
            <Paper sx={{ p: 3, height: '400px' }}>
              <Typography variant="h6" gutterBottom>
                Employee Growth Trend
              </Typography>
              <ResponsiveContainer width="100%" height="85%">
                <AreaChart data={employeeGrowthData}>
                  <CartesianGrid strokeDasharray="3 3" />
                  <XAxis dataKey="month" />
                  <YAxis />
                  <Tooltip />
                  <Area type="monotone" dataKey="employees" stroke="#1976d2" fill="#1976d2" fillOpacity={0.3} />
                  <Bar dataKey="hires" fill="#4caf50" />
                  <Bar dataKey="terminations" fill="#f44336" />
                </AreaChart>
              </ResponsiveContainer>
            </Paper>
          </Grid>
          <Grid
            size={{
              xs: 12,
              md: 4
            }}>
            <Paper sx={{ p: 3, height: '400px' }}>
              <Typography variant="h6" gutterBottom>
                Department Distribution
              </Typography>
              <ResponsiveContainer width="100%" height="85%">
                <PieChart>
                  <Pie
                    data={departmentSalaryData}
                    cx="50%"
                    cy="50%"
                    labelLine={false}
                    label={({ name, employees }) => `${name} (${employees})`}
                    outerRadius={80}
                    fill="#8884d8"
                    dataKey="employees"
                  >
                    {departmentSalaryData.map((entry, index) => (
                      <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                    ))}
                  </Pie>
                  <Tooltip />
                </PieChart>
              </ResponsiveContainer>
            </Paper>
          </Grid>
          <Grid size={12}>
            <Paper sx={{ p: 3 }}>
              <Typography variant="h6" gutterBottom>
                Top Performers
              </Typography>
              <TableContainer>
                <Table>
                  <TableHead>
                    <TableRow>
                      <TableCell>Employee</TableCell>
                      <TableCell>Department</TableCell>
                      <TableCell>Performance Score</TableCell>
                      <TableCell>Salary</TableCell>
                      <TableCell>Status</TableCell>
                    </TableRow>
                  </TableHead>
                  <TableBody>
                    {topPerformers.map((employee, index) => (
                      <TableRow key={index}>
                        <TableCell>{employee.name}</TableCell>
                        <TableCell>{employee.department}</TableCell>
                        <TableCell>
                          <Chip
                            label={`${employee.performance}%`}
                            color={employee.performance >= 95 ? 'success' : employee.performance >= 90 ? 'warning' : 'default'}
                            size="small"
                          />
                        </TableCell>
                        <TableCell>${employee.salary.toLocaleString()}</TableCell>
                        <TableCell>
                          <Chip label="Active" color="success" size="small" />
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </TableContainer>
            </Paper>
          </Grid>
        </Grid>
      </TabPanel>
      <TabPanel value={tabValue} index={1}>
        {/* Attendance Reports */}
        <Grid container spacing={3}>
          <Grid
            size={{
              xs: 12,
              md: 8
            }}>
            <Paper sx={{ p: 3, height: '400px' }}>
              <Typography variant="h6" gutterBottom>
                Department-wise Attendance
              </Typography>
              <ResponsiveContainer width="100%" height="85%">
                <BarChart data={attendanceData}>
                  <CartesianGrid strokeDasharray="3 3" />
                  <XAxis dataKey="department" />
                  <YAxis />
                  <Tooltip />
                  <Bar dataKey="present" fill="#4caf50" name="Present" />
                  <Bar dataKey="late" fill="#ff9800" name="Late" />
                  <Bar dataKey="absent" fill="#f44336" name="Absent" />
                </BarChart>
              </ResponsiveContainer>
            </Paper>
          </Grid>
          <Grid
            size={{
              xs: 12,
              md: 4
            }}>
            <Grid container spacing={2}>
              <Grid size={12}>
                <Card>
                  <CardContent sx={{ textAlign: 'center' }}>
                    <Typography variant="h4" color="success.main">
                      92%
                    </Typography>
                    <Typography color="textSecondary">
                      Average Attendance Rate
                    </Typography>
                  </CardContent>
                </Card>
              </Grid>
              <Grid size={12}>
                <Card>
                  <CardContent sx={{ textAlign: 'center' }}>
                    <Typography variant="h4" color="warning.main">
                      7
                    </Typography>
                    <Typography color="textSecondary">
                      Late Arrivals Today
                    </Typography>
                  </CardContent>
                </Card>
              </Grid>
              <Grid size={12}>
                <Card>
                  <CardContent sx={{ textAlign: 'center' }}>
                    <Typography variant="h4" color="error.main">
                      12
                    </Typography>
                    <Typography color="textSecondary">
                      Absent Today
                    </Typography>
                  </CardContent>
                </Card>
              </Grid>
            </Grid>
          </Grid>
        </Grid>
      </TabPanel>
      <TabPanel value={tabValue} index={2}>
        {/* Payroll Analysis */}
        <Grid container spacing={3}>
          <Grid
            size={{
              xs: 12,
              md: 8
            }}>
            <Paper sx={{ p: 3, height: '400px' }}>
              <Typography variant="h6" gutterBottom>
                Monthly Payroll Trend
              </Typography>
              <ResponsiveContainer width="100%" height="85%">
                <AreaChart data={payrollTrend}>
                  <CartesianGrid strokeDasharray="3 3" />
                  <XAxis dataKey="month" />
                  <YAxis />
                  <Tooltip formatter={(value) => [`$${value.toLocaleString()}`, 'Payroll']} />
                  <Area type="monotone" dataKey="amount" stroke="#1976d2" fill="#1976d2" fillOpacity={0.6} />
                </AreaChart>
              </ResponsiveContainer>
            </Paper>
          </Grid>
          <Grid
            size={{
              xs: 12,
              md: 4
            }}>
            <Paper sx={{ p: 3, height: '400px' }}>
              <Typography variant="h6" gutterBottom>
                Department Salary Distribution
              </Typography>
              <ResponsiveContainer width="100%" height="85%">
                <PieChart>
                  <Pie
                    data={departmentSalaryData}
                    cx="50%"
                    cy="50%"
                    labelLine={false}
                    label={({ name, value }) => `${name} $${((value || 0)/1000).toFixed(0)}K`}
                    outerRadius={80}
                    fill="#8884d8"
                    dataKey="value"
                  >
                    {departmentSalaryData.map((entry, index) => (
                      <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                    ))}
                  </Pie>
                  <Tooltip formatter={(value) => `$${value.toLocaleString()}`} />
                </PieChart>
              </ResponsiveContainer>
            </Paper>
          </Grid>
        </Grid>
      </TabPanel>
      <TabPanel value={tabValue} index={3}>
        {/* Leave Reports */}
        <Grid container spacing={3}>
          <Grid size={12}>
            <Paper sx={{ p: 3, height: '400px' }}>
              <Typography variant="h6" gutterBottom>
                Leave Trends by Type
              </Typography>
              <ResponsiveContainer width="100%" height="85%">
                <LineChart data={leaveAnalytics}>
                  <CartesianGrid strokeDasharray="3 3" />
                  <XAxis dataKey="month" />
                  <YAxis />
                  <Tooltip />
                  <Line type="monotone" dataKey="sick" stroke="#f44336" strokeWidth={2} name="Sick Leave" />
                  <Line type="monotone" dataKey="vacation" stroke="#2196f3" strokeWidth={2} name="Vacation" />
                  <Line type="monotone" dataKey="personal" stroke="#ff9800" strokeWidth={2} name="Personal" />
                  <Line type="monotone" dataKey="total" stroke="#4caf50" strokeWidth={3} name="Total" />
                </LineChart>
              </ResponsiveContainer>
            </Paper>
          </Grid>
          <Grid
            size={{
              xs: 12,
              md: 4
            }}>
            <Card>
              <CardContent sx={{ textAlign: 'center' }}>
                <Typography variant="h4" color="primary.main">
                  238
                </Typography>
                <Typography color="textSecondary">
                  Total Leave Days (6 months)
                </Typography>
              </CardContent>
            </Card>
          </Grid>
          <Grid
            size={{
              xs: 12,
              md: 4
            }}>
            <Card>
              <CardContent sx={{ textAlign: 'center' }}>
                <Typography variant="h4" color="warning.main">
                  12
                </Typography>
                <Typography color="textSecondary">
                  Pending Approvals
                </Typography>
              </CardContent>
            </Card>
          </Grid>
          <Grid
            size={{
              xs: 12,
              md: 4
            }}>
            <Card>
              <CardContent sx={{ textAlign: 'center' }}>
                <Typography variant="h4" color="success.main">
                  1.9
                </Typography>
                <Typography color="textSecondary">
                  Avg Days per Employee
                </Typography>
              </CardContent>
            </Card>
          </Grid>
        </Grid>
      </TabPanel>
      <TabPanel value={tabValue} index={4}>
        {/* Performance Reports */}
        <Grid container spacing={3}>
          <Grid size={12}>
            <Paper sx={{ p: 3 }}>
              <Typography variant="h6" gutterBottom>
                Performance Summary
              </Typography>
              <Typography variant="body1" color="textSecondary">
                Detailed performance analytics and KPI tracking will be implemented here.
                This section will include employee performance metrics, goal tracking,
                and comparative analysis across departments.
              </Typography>
            </Paper>
          </Grid>
        </Grid>
      </TabPanel>
    </Box>
  );
};