import { LeaveRequest } from '../types';
import { format, differenceInDays, parseISO } from 'date-fns';
import { mailService } from './mailService';
import { employeeService } from './employeeService';

// Mock data - replace with actual API calls
const mockLeaves: LeaveRequest[] = [
  {
    id: '1',
    employeeId: '1',
    type: 'vacation',
    startDate: '2023-06-25',
    endDate: '2023-06-30',
    days: 6,
    reason: 'Summer vacation with family',
    status: 'approved',
    approvedBy: 'HR001',
    appliedDate: '2023-06-15',
    comments: 'Approved for summer break'
  },
  {
    id: '2',
    employeeId: '2',
    type: 'sick',
    startDate: '2023-06-20',
    endDate: '2023-06-21',
    days: 2,
    reason: 'Medical appointment and recovery',
    status: 'pending',
    appliedDate: '2023-06-19',
  },
  {
    id: '3',
    employeeId: '3',
    type: 'personal',
    startDate: '2023-07-01',
    endDate: '2023-07-01',
    days: 1,
    reason: 'Personal matters',
    status: 'rejected',
    approvedBy: 'HR001',
    appliedDate: '2023-06-18',
    comments: 'Conflicting project deadline'
  },
  {
    id: '4',
    employeeId: '4',
    type: 'maternity',
    startDate: '2023-07-15',
    endDate: '2023-10-15',
    days: 90,
    reason: 'Maternity leave',
    status: 'approved',
    approvedBy: 'HR001',
    appliedDate: '2023-06-10',
    comments: 'Standard maternity leave approved'
  },
];

let leaveRequests = [...mockLeaves];

export const leaveService = {
  // Get all leave requests
  getAllLeaves: async (): Promise<LeaveRequest[]> => {
    return new Promise((resolve) => {
      setTimeout(() => resolve([...leaveRequests]), 500);
    });
  },

  // Get leave request by ID
  getLeaveById: async (id: string): Promise<LeaveRequest | null> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const leave = leaveRequests.find(req => req.id === id);
        resolve(leave || null);
      }, 300);
    });
  },

  // Get leaves by employee ID
  getLeavesByEmployeeId: async (employeeId: string): Promise<LeaveRequest[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const filtered = leaveRequests.filter(req => req.employeeId === employeeId);
        resolve(filtered);
      }, 300);
    });
  },

  // Get pending leave requests
  getPendingLeaves: async (): Promise<LeaveRequest[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const pending = leaveRequests.filter(req => req.status === 'pending');
        resolve(pending);
      }, 300);
    });
  },

  // Create new leave request
  createLeaveRequest: async (leaveData: Omit<LeaveRequest, 'id' | 'appliedDate' | 'status'>): Promise<LeaveRequest> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const days = differenceInDays(parseISO(leaveData.endDate), parseISO(leaveData.startDate)) + 1;
        
        const newLeave: LeaveRequest = {
          ...leaveData,
          id: Date.now().toString(),
          days: days > 0 ? days : 1,
          appliedDate: format(new Date(), 'yyyy-MM-dd'),
          status: 'pending',
        };
        
        leaveRequests.push(newLeave);
        resolve(newLeave);
      }, 500);
    });
  },

  // Update leave request
  updateLeaveRequest: async (id: string, leaveData: Partial<LeaveRequest>): Promise<LeaveRequest | null> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const index = leaveRequests.findIndex(req => req.id === id);
        if (index !== -1) {
          // Recalculate days if dates are updated
          if (leaveData.startDate || leaveData.endDate) {
            const startDate = leaveData.startDate || leaveRequests[index].startDate;
            const endDate = leaveData.endDate || leaveRequests[index].endDate;
            const days = differenceInDays(parseISO(endDate), parseISO(startDate)) + 1;
            leaveData.days = days > 0 ? days : 1;
          }
          
          leaveRequests[index] = { ...leaveRequests[index], ...leaveData };
          resolve(leaveRequests[index]);
        } else {
          resolve(null);
        }
      }, 500);
    });
  },

  // Delete leave request
  deleteLeaveRequest: async (id: string): Promise<boolean> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const index = leaveRequests.findIndex(req => req.id === id);
        if (index !== -1) {
          leaveRequests.splice(index, 1);
          resolve(true);
        } else {
          resolve(false);
        }
      }, 300);
    });
  },

  // Approve leave request
  approveLeave: async (id: string, approvedBy: string, comments?: string): Promise<LeaveRequest | null> => {
    return new Promise(async (resolve) => {
      setTimeout(async () => {
        const index = leaveRequests.findIndex(req => req.id === id);
        if (index !== -1) {
          leaveRequests[index] = {
            ...leaveRequests[index],
            status: 'approved',
            approvedBy,
            comments,
          };
          
          // Send approval email
          try {
            const employee = await employeeService.getEmployeeById(leaveRequests[index].employeeId);
            if (employee) {
              await mailService.sendLeaveApprovalEmail(
                employee.email,
                `${employee.firstName} ${employee.lastName}`,
                format(parseISO(leaveRequests[index].startDate), 'MMM dd, yyyy'),
                format(parseISO(leaveRequests[index].endDate), 'MMM dd, yyyy')
              );
            }
          } catch (error) {
            console.error('Failed to send leave approval email:', error);
          }
          
          resolve(leaveRequests[index]);
        } else {
          resolve(null);
        }
      }, 500);
    });
  },

  // Reject leave request
  rejectLeave: async (id: string, rejectedBy: string, comments?: string): Promise<LeaveRequest | null> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const index = leaveRequests.findIndex(req => req.id === id);
        if (index !== -1) {
          leaveRequests[index] = {
            ...leaveRequests[index],
            status: 'rejected',
            approvedBy: rejectedBy,
            comments,
          };
          resolve(leaveRequests[index]);
        } else {
          resolve(null);
        }
      }, 500);
    });
  },

  // Get leave statistics
  getLeaveStats: async (): Promise<{
    total: number;
    pending: number;
    approved: number;
    rejected: number;
    totalApprovedDays: number;
    byType: {
      [key: string]: number;
    };
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const stats = {
          total: leaveRequests.length,
          pending: leaveRequests.filter(req => req.status === 'pending').length,
          approved: leaveRequests.filter(req => req.status === 'approved').length,
          rejected: leaveRequests.filter(req => req.status === 'rejected').length,
          totalApprovedDays: leaveRequests
            .filter(req => req.status === 'approved')
            .reduce((sum, req) => sum + req.days, 0),
          byType: {} as { [key: string]: number },
        };

        // Group by leave type
        leaveRequests.forEach(req => {
          if (!stats.byType[req.type]) {
            stats.byType[req.type] = 0;
          }
          stats.byType[req.type]++;
        });

        resolve(stats);
      }, 300);
    });
  },

  // Get employee leave balance
  getEmployeeLeaveBalance: async (employeeId: string): Promise<{
    employeeId: string;
    totalEntitled: {
      sick: number;
      vacation: number;
      personal: number;
    };
    used: {
      sick: number;
      vacation: number;
      personal: number;
    };
    remaining: {
      sick: number;
      vacation: number;
      personal: number;
    };
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const employeeLeaves = leaveRequests.filter(
          req => req.employeeId === employeeId && req.status === 'approved'
        );

        const currentYear = new Date().getFullYear();
        const yearlyLeaves = employeeLeaves.filter(
          req => parseISO(req.startDate).getFullYear() === currentYear
        );

        const used = {
          sick: yearlyLeaves.filter(req => req.type === 'sick').reduce((sum, req) => sum + req.days, 0),
          vacation: yearlyLeaves.filter(req => req.type === 'vacation').reduce((sum, req) => sum + req.days, 0),
          personal: yearlyLeaves.filter(req => req.type === 'personal').reduce((sum, req) => sum + req.days, 0),
        };

        // Standard entitlements (can be configured per employee)
        const totalEntitled = {
          sick: 10,
          vacation: 20,
          personal: 5,
        };

        const remaining = {
          sick: Math.max(0, totalEntitled.sick - used.sick),
          vacation: Math.max(0, totalEntitled.vacation - used.vacation),
          personal: Math.max(0, totalEntitled.personal - used.personal),
        };

        resolve({
          employeeId,
          totalEntitled,
          used,
          remaining,
        });
      }, 500);
    });
  },

  // Get leaves by date range
  getLeavesByDateRange: async (startDate: string, endDate: string): Promise<LeaveRequest[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const filtered = leaveRequests.filter(req => {
          const reqStart = parseISO(req.startDate);
          const reqEnd = parseISO(req.endDate);
          const rangeStart = parseISO(startDate);
          const rangeEnd = parseISO(endDate);
          
          // Check if leave period overlaps with the query range
          return (reqStart <= rangeEnd && reqEnd >= rangeStart);
        });
        resolve(filtered);
      }, 300);
    });
  },

  // Check for leave conflicts
  checkLeaveConflicts: async (employeeId: string, startDate: string, endDate: string, excludeId?: string): Promise<LeaveRequest[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const conflicts = leaveRequests.filter(req => {
          if (excludeId && req.id === excludeId) return false;
          if (req.employeeId !== employeeId) return false;
          if (req.status === 'rejected') return false;

          const reqStart = parseISO(req.startDate);
          const reqEnd = parseISO(req.endDate);
          const newStart = parseISO(startDate);
          const newEnd = parseISO(endDate);
          
          // Check for overlap
          return (reqStart <= newEnd && reqEnd >= newStart);
        });
        resolve(conflicts);
      }, 300);
    });
  },

  // Get leave calendar data
  getLeaveCalendar: async (month: number, year: number): Promise<Array<{
    date: string;
    leaves: Array<{
      employeeId: string;
      employeeName: string;
      type: string;
      status: string;
    }>;
  }>> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const monthStart = new Date(year, month - 1, 1);
        const monthEnd = new Date(year, month, 0);
        
        const calendar: { [key: string]: any } = {};
        
        leaveRequests.forEach(req => {
          if (req.status === 'rejected') return;
          
          const startDate = parseISO(req.startDate);
          const endDate = parseISO(req.endDate);
          
          // Generate all dates in the leave period
          for (let d = new Date(startDate); d <= endDate; d.setDate(d.getDate() + 1)) {
            if (d >= monthStart && d <= monthEnd) {
              const dateStr = format(d, 'yyyy-MM-dd');
              if (!calendar[dateStr]) {
                calendar[dateStr] = {
                  date: dateStr,
                  leaves: [],
                };
              }
              
              calendar[dateStr].leaves.push({
                employeeId: req.employeeId,
                employeeName: `Employee ${req.employeeId}`, // Would get from employee service
                type: req.type,
                status: req.status,
              });
            }
          }
        });

        resolve(Object.values(calendar));
      }, 500);
    });
  }
};