import { PayrollRecord } from '../types';
import { format } from 'date-fns';
import { mailService } from './mailService';
import { employeeService } from './employeeService';

// Mock data - replace with actual API calls
const mockPayrollRecords: PayrollRecord[] = [
  {
    id: '1',
    employeeId: '1',
    month: 'June',
    year: 2023,
    basicSalary: 8000,
    allowances: 1500,
    deductions: 800,
    overtime: 500,
    netSalary: 9200,
    payDate: '2023-06-30',
    status: 'paid',
  },
  {
    id: '2',
    employeeId: '2',
    month: 'June',
    year: 2023,
    basicSalary: 6500,
    allowances: 1000,
    deductions: 650,
    overtime: 300,
    netSalary: 7150,
    payDate: '2023-06-30',
    status: 'paid',
  },
  {
    id: '3',
    employeeId: '3',
    month: 'June',
    year: 2023,
    basicSalary: 5500,
    allowances: 800,
    deductions: 550,
    overtime: 200,
    netSalary: 5950,
    payDate: '2023-06-30',
    status: 'processed',
  },
  {
    id: '4',
    employeeId: '4',
    month: 'June',
    year: 2023,
    basicSalary: 5000,
    allowances: 750,
    deductions: 500,
    overtime: 150,
    netSalary: 5400,
    payDate: '2023-06-30',
    status: 'processed',
  },
  {
    id: '5',
    employeeId: '5',
    month: 'June',
    year: 2023,
    basicSalary: 6000,
    allowances: 900,
    deductions: 600,
    overtime: 250,
    netSalary: 6550,
    payDate: '',
    status: 'draft',
  },
];

let payrollRecords = [...mockPayrollRecords];

export const payrollService = {
  // Get all payroll records
  getAllPayroll: async (): Promise<PayrollRecord[]> => {
    return new Promise((resolve) => {
      setTimeout(() => resolve([...payrollRecords]), 500);
    });
  },

  // Get payroll record by ID
  getPayrollById: async (id: string): Promise<PayrollRecord | null> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const record = payrollRecords.find(rec => rec.id === id);
        resolve(record || null);
      }, 300);
    });
  },

  // Get payroll by employee ID
  getPayrollByEmployeeId: async (employeeId: string): Promise<PayrollRecord[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const filtered = payrollRecords.filter(rec => rec.employeeId === employeeId);
        resolve(filtered);
      }, 300);
    });
  },

  // Get payroll by month and year
  getPayrollByPeriod: async (month: string, year: number): Promise<PayrollRecord[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const filtered = payrollRecords.filter(
          rec => rec.month === month && rec.year === year
        );
        resolve(filtered);
      }, 300);
    });
  },

  // Get payroll by status
  getPayrollByStatus: async (status: 'draft' | 'processed' | 'paid'): Promise<PayrollRecord[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const filtered = payrollRecords.filter(rec => rec.status === status);
        resolve(filtered);
      }, 300);
    });
  },

  // Create payroll record
  createPayroll: async (payrollData: Omit<PayrollRecord, 'id' | 'netSalary'>): Promise<PayrollRecord> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const netSalary = payrollData.basicSalary + 
                         payrollData.allowances + 
                         payrollData.overtime - 
                         payrollData.deductions;

        const newPayroll: PayrollRecord = {
          ...payrollData,
          id: Date.now().toString(),
          netSalary,
        };
        
        payrollRecords.push(newPayroll);
        resolve(newPayroll);
      }, 500);
    });
  },

  // Update payroll record
  updatePayroll: async (id: string, payrollData: Partial<PayrollRecord>): Promise<PayrollRecord | null> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const index = payrollRecords.findIndex(rec => rec.id === id);
        if (index !== -1) {
          const updatedRecord = { ...payrollRecords[index], ...payrollData };
          
          // Recalculate net salary if components changed
          if (payrollData.basicSalary !== undefined || 
              payrollData.allowances !== undefined || 
              payrollData.overtime !== undefined || 
              payrollData.deductions !== undefined) {
            updatedRecord.netSalary = updatedRecord.basicSalary + 
                                     updatedRecord.allowances + 
                                     updatedRecord.overtime - 
                                     updatedRecord.deductions;
          }
          
          payrollRecords[index] = updatedRecord;
          resolve(updatedRecord);
        } else {
          resolve(null);
        }
      }, 500);
    });
  },

  // Delete payroll record
  deletePayroll: async (id: string): Promise<boolean> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const index = payrollRecords.findIndex(rec => rec.id === id);
        if (index !== -1) {
          payrollRecords.splice(index, 1);
          resolve(true);
        } else {
          resolve(false);
        }
      }, 300);
    });
  },

  // Process payroll (draft -> processed)
  processPayroll: async (id: string): Promise<PayrollRecord | null> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const index = payrollRecords.findIndex(rec => rec.id === id);
        if (index !== -1 && payrollRecords[index].status === 'draft') {
          payrollRecords[index].status = 'processed';
          resolve(payrollRecords[index]);
        } else {
          resolve(null);
        }
      }, 500);
    });
  },

  // Mark as paid (processed -> paid)
  markAsPaid: async (id: string): Promise<PayrollRecord | null> => {
    return new Promise(async (resolve) => {
      setTimeout(async () => {
        const index = payrollRecords.findIndex(rec => rec.id === id);
        if (index !== -1 && payrollRecords[index].status === 'processed') {
          payrollRecords[index].status = 'paid';
          payrollRecords[index].payDate = format(new Date(), 'yyyy-MM-dd');
          
          // Send payroll notification email
          try {
            const employee = await employeeService.getEmployeeById(payrollRecords[index].employeeId);
            if (employee) {
              await mailService.sendPayrollNotification(
                employee.email,
                `${employee.firstName} ${employee.lastName}`,
                `${payrollRecords[index].month} ${payrollRecords[index].year}`
              );
            }
          } catch (error) {
            console.error('Failed to send payroll notification email:', error);
          }
          
          resolve(payrollRecords[index]);
        } else {
          resolve(null);
        }
      }, 500);
    });
  },

  // Bulk process payroll
  bulkProcessPayroll: async (ids: string[]): Promise<PayrollRecord[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const processed: PayrollRecord[] = [];
        
        ids.forEach(id => {
          const index = payrollRecords.findIndex(rec => rec.id === id);
          if (index !== -1 && payrollRecords[index].status === 'draft') {
            payrollRecords[index].status = 'processed';
            processed.push(payrollRecords[index]);
          }
        });
        
        resolve(processed);
      }, 1000);
    });
  },

  // Bulk mark as paid
  bulkMarkAsPaid: async (ids: string[]): Promise<PayrollRecord[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const paid: PayrollRecord[] = [];
        const payDate = format(new Date(), 'yyyy-MM-dd');
        
        ids.forEach(id => {
          const index = payrollRecords.findIndex(rec => rec.id === id);
          if (index !== -1 && payrollRecords[index].status === 'processed') {
            payrollRecords[index].status = 'paid';
            payrollRecords[index].payDate = payDate;
            paid.push(payrollRecords[index]);
          }
        });
        
        resolve(paid);
      }, 1000);
    });
  },

  // Get payroll statistics
  getPayrollStats: async (month?: string, year?: number): Promise<{
    total: number;
    draft: number;
    processed: number;
    paid: number;
    totalAmount: number;
    pendingAmount: number;
    paidAmount: number;
    averageSalary: number;
    totalBasicSalary: number;
    totalAllowances: number;
    totalDeductions: number;
    totalOvertime: number;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        let filteredRecords = payrollRecords;
        
        if (month && year) {
          filteredRecords = payrollRecords.filter(
            rec => rec.month === month && rec.year === year
          );
        }

        const stats = {
          total: filteredRecords.length,
          draft: filteredRecords.filter(rec => rec.status === 'draft').length,
          processed: filteredRecords.filter(rec => rec.status === 'processed').length,
          paid: filteredRecords.filter(rec => rec.status === 'paid').length,
          totalAmount: filteredRecords.reduce((sum, rec) => sum + rec.netSalary, 0),
          pendingAmount: filteredRecords
            .filter(rec => rec.status === 'processed')
            .reduce((sum, rec) => sum + rec.netSalary, 0),
          paidAmount: filteredRecords
            .filter(rec => rec.status === 'paid')
            .reduce((sum, rec) => sum + rec.netSalary, 0),
          averageSalary: 0,
          totalBasicSalary: filteredRecords.reduce((sum, rec) => sum + rec.basicSalary, 0),
          totalAllowances: filteredRecords.reduce((sum, rec) => sum + rec.allowances, 0),
          totalDeductions: filteredRecords.reduce((sum, rec) => sum + rec.deductions, 0),
          totalOvertime: filteredRecords.reduce((sum, rec) => sum + rec.overtime, 0),
        };

        stats.averageSalary = stats.total > 0 ? stats.totalAmount / stats.total : 0;

        resolve(stats);
      }, 300);
    });
  },

  // Generate monthly payroll for all employees
  generateMonthlyPayroll: async (
    month: string, 
    year: number, 
    employeeData: Array<{
      employeeId: string;
      basicSalary: number;
      allowances: number;
      deductions: number;
      overtimeHours?: number;
      overtimeRate?: number;
    }>
  ): Promise<PayrollRecord[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const newRecords: PayrollRecord[] = [];
        
        employeeData.forEach(emp => {
          const overtime = (emp.overtimeHours || 0) * (emp.overtimeRate || 25); // Default $25/hour
          const netSalary = emp.basicSalary + emp.allowances + overtime - emp.deductions;
          
          const record: PayrollRecord = {
            id: `${Date.now()}_${emp.employeeId}`,
            employeeId: emp.employeeId,
            month,
            year,
            basicSalary: emp.basicSalary,
            allowances: emp.allowances,
            deductions: emp.deductions,
            overtime,
            netSalary,
            payDate: '',
            status: 'draft',
          };
          
          newRecords.push(record);
        });
        
        payrollRecords.push(...newRecords);
        resolve(newRecords);
      }, 1000);
    });
  },

  // Calculate payroll from attendance
  calculatePayrollFromAttendance: async (
    employeeId: string, 
    month: string, 
    year: number, 
    baseSalary: number,
    attendanceData: {
      workingDays: number;
      overtimeHours: number;
      totalWorkingHours: number;
    }
  ): Promise<{
    basicSalary: number;
    overtime: number;
    deductions: number;
    netSalary: number;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const standardWorkingDays = 22; // Assumed standard working days
        const dailyRate = baseSalary / standardWorkingDays;
        
        // Calculate pro-rated salary based on attendance
        const basicSalary = dailyRate * attendanceData.workingDays;
        
        // Calculate overtime (assuming $25/hour overtime rate)
        const overtime = attendanceData.overtimeHours * 25;
        
        // Calculate deductions (absent days)
        const absentDays = Math.max(0, standardWorkingDays - attendanceData.workingDays);
        const deductions = dailyRate * absentDays;
        
        const netSalary = basicSalary + overtime - deductions;
        
        resolve({
          basicSalary,
          overtime,
          deductions,
          netSalary,
        });
      }, 500);
    });
  },

  // Get employee payroll history
  getEmployeePayrollHistory: async (
    employeeId: string, 
    startYear?: number, 
    endYear?: number
  ): Promise<{
    employeeId: string;
    totalRecords: number;
    totalPaid: number;
    averageMonthly: number;
    yearlyBreakdown: Array<{
      year: number;
      totalAmount: number;
      records: number;
    }>;
    records: PayrollRecord[];
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        let filtered = payrollRecords.filter(rec => rec.employeeId === employeeId);
        
        if (startYear && endYear) {
          filtered = filtered.filter(rec => rec.year >= startYear && rec.year <= endYear);
        }
        
        const totalPaid = filtered
          .filter(rec => rec.status === 'paid')
          .reduce((sum, rec) => sum + rec.netSalary, 0);
        
        // Group by year
        const yearlyMap: { [year: number]: { totalAmount: number; records: number } } = {};
        filtered.forEach(rec => {
          if (!yearlyMap[rec.year]) {
            yearlyMap[rec.year] = { totalAmount: 0, records: 0 };
          }
          yearlyMap[rec.year].totalAmount += rec.netSalary;
          yearlyMap[rec.year].records++;
        });
        
        const yearlyBreakdown = Object.entries(yearlyMap).map(([year, data]) => ({
          year: parseInt(year),
          totalAmount: data.totalAmount,
          records: data.records,
        }));
        
        resolve({
          employeeId,
          totalRecords: filtered.length,
          totalPaid,
          averageMonthly: filtered.length > 0 ? totalPaid / filtered.length : 0,
          yearlyBreakdown,
          records: filtered.sort((a, b) => b.year - a.year || b.month.localeCompare(a.month)),
        });
      }, 500);
    });
  },

  // Get payroll summary for reporting
  getPayrollSummary: async (startMonth: string, startYear: number, endMonth: string, endYear: number): Promise<{
    period: string;
    totalRecords: number;
    totalAmount: number;
    monthlyBreakdown: Array<{
      month: string;
      year: number;
      totalAmount: number;
      recordCount: number;
      averageSalary: number;
    }>;
    departmentBreakdown: Array<{
      department: string;
      totalAmount: number;
      employeeCount: number;
      averageSalary: number;
    }>;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        // This would typically involve joining with employee data to get department info
        const monthlyMap: { [key: string]: { totalAmount: number; recordCount: number } } = {};
        
        const filtered = payrollRecords.filter(rec => {
          const recDate = new Date(rec.year, getMonthNumber(rec.month) - 1);
          const startDate = new Date(startYear, getMonthNumber(startMonth) - 1);
          const endDate = new Date(endYear, getMonthNumber(endMonth) - 1);
          return recDate >= startDate && recDate <= endDate;
        });
        
        filtered.forEach(rec => {
          const key = `${rec.month}_${rec.year}`;
          if (!monthlyMap[key]) {
            monthlyMap[key] = { totalAmount: 0, recordCount: 0 };
          }
          monthlyMap[key].totalAmount += rec.netSalary;
          monthlyMap[key].recordCount++;
        });
        
        const monthlyBreakdown = Object.entries(monthlyMap).map(([key, data]) => {
          const [month, year] = key.split('_');
          return {
            month,
            year: parseInt(year),
            totalAmount: data.totalAmount,
            recordCount: data.recordCount,
            averageSalary: data.totalAmount / data.recordCount,
          };
        });
        
        // Mock department breakdown (would need employee service integration)
        const departmentBreakdown = [
          { department: 'Engineering', totalAmount: 450000, employeeCount: 15, averageSalary: 30000 },
          { department: 'Sales', totalAmount: 320000, employeeCount: 12, averageSalary: 26667 },
          { department: 'Marketing', totalAmount: 280000, employeeCount: 10, averageSalary: 28000 },
        ];
        
        resolve({
          period: `${startMonth} ${startYear} - ${endMonth} ${endYear}`,
          totalRecords: filtered.length,
          totalAmount: filtered.reduce((sum, rec) => sum + rec.netSalary, 0),
          monthlyBreakdown,
          departmentBreakdown,
        });
      }, 500);
    });
  }
};

// Helper function to convert month name to number
function getMonthNumber(monthName: string): number {
  const months = [
    'January', 'February', 'March', 'April', 'May', 'June',
    'July', 'August', 'September', 'October', 'November', 'December'
  ];
  return months.indexOf(monthName) + 1 || 1;
}