import { 
  JobPosting, 
  Candidate, 
  Interview, 
  OfferLetter, 
  OnboardingProcess, 
  RecruitmentMetrics,
  JobPostingRequest,
  CandidateRequest,
  InterviewRequest,
  OnboardingProcessRequest,
  ApplicationHistory,
  InterviewFeedback,
  OnboardingTask
} from '../types/recruitment';

const mockJobPostings: JobPosting[] = [
  {
    id: 'job-001',
    title: 'Senior Software Engineer',
    department: 'Engineering',
    location: 'Bangalore, India',
    employmentType: 'Full-time',
    experienceLevel: 'Senior Level',
    salaryRange: { min: 1200000, max: 1800000, currency: 'INR' },
    description: 'We are looking for a Senior Software Engineer to join our dynamic engineering team...',
    requirements: ['5+ years of experience', 'React/Node.js', 'System Design', 'Leadership skills'],
    responsibilities: ['Lead development projects', 'Mentor junior developers', 'Code reviews'],
    benefits: ['Health Insurance', 'Flexible Working', 'Stock Options', 'Learning Budget'],
    postedDate: '2024-01-15',
    applicationDeadline: '2024-02-15',
    status: 'Published',
    hiringManager: 'John Smith',
    applicationCount: 25,
    createdBy: 'hr@company.com',
    updatedAt: '2024-01-15T10:00:00Z'
  },
  {
    id: 'job-002',
    title: 'Product Manager',
    department: 'Product',
    location: 'Mumbai, India',
    employmentType: 'Full-time',
    experienceLevel: 'Mid Level',
    salaryRange: { min: 1500000, max: 2200000, currency: 'INR' },
    description: 'Join our product team to drive innovation and user experience...',
    requirements: ['3+ years PM experience', 'Analytics', 'User Research', 'Strategy'],
    responsibilities: ['Product roadmap', 'Stakeholder management', 'User research'],
    benefits: ['Health Insurance', 'Remote Work', 'Performance Bonus', 'Professional Development'],
    postedDate: '2024-01-20',
    applicationDeadline: '2024-03-01',
    status: 'Published',
    hiringManager: 'Sarah Johnson',
    applicationCount: 18,
    createdBy: 'hr@company.com',
    updatedAt: '2024-01-20T09:30:00Z'
  }
];

const mockCandidates: Candidate[] = [
  {
    id: 'candidate-001',
    firstName: 'Rahul',
    lastName: 'Sharma',
    email: 'rahul.sharma@email.com',
    phone: '+91-9876543210',
    address: 'Bangalore, Karnataka',
    resumeUrl: '/uploads/rahul-sharma-resume.pdf',
    linkedInProfile: 'https://linkedin.com/in/rahul-sharma',
    currentPosition: 'Software Engineer',
    currentCompany: 'Tech Solutions Pvt Ltd',
    totalExperience: 4,
    expectedSalary: 1500000,
    noticePeriod: '60 days',
    skills: ['React', 'Node.js', 'MongoDB', 'AWS', 'Docker'],
    education: [{
      id: 'edu-001',
      degree: 'B.Tech Computer Science',
      institution: 'IIT Delhi',
      fieldOfStudy: 'Computer Science',
      startDate: '2018-07-01',
      endDate: '2022-06-30',
      grade: '8.5 CGPA'
    }],
    workExperience: [{
      id: 'exp-001',
      jobTitle: 'Software Engineer',
      company: 'Tech Solutions Pvt Ltd',
      location: 'Bangalore',
      startDate: '2022-07-01',
      isCurrentJob: true,
      description: 'Full-stack development using MERN stack',
      achievements: ['Led team of 3 developers', 'Reduced load time by 40%']
    }],
    appliedJobs: [{
      id: 'app-001',
      jobId: 'job-001',
      jobTitle: 'Senior Software Engineer',
      appliedDate: '2024-01-16',
      status: 'Under Review',
      stage: 'Resume Screening',
      interviewHistory: []
    }],
    status: 'Reviewed',
    source: 'LinkedIn',
    notes: 'Strong technical background, good communication skills',
    createdAt: '2024-01-16T14:30:00Z',
    updatedAt: '2024-01-17T10:15:00Z'
  }
];

const mockInterviews: Interview[] = [
  {
    id: 'interview-001',
    candidateId: 'candidate-001',
    jobId: 'job-001',
    type: 'Technical',
    round: 1,
    scheduledDate: '2024-01-25T10:00:00Z',
    duration: 60,
    meetingLink: 'https://meet.google.com/abc-def-ghi',
    interviewers: [{
      id: 'int-001',
      name: 'John Smith',
      email: 'john.smith@company.com',
      role: 'Senior Engineering Manager',
      department: 'Engineering',
      isLead: true
    }],
    status: 'Scheduled',
    createdBy: 'hr@company.com',
    createdAt: '2024-01-18T11:00:00Z',
    updatedAt: '2024-01-18T11:00:00Z'
  }
];

const mockOfferLetters: OfferLetter[] = [];

const mockOnboardingProcesses: OnboardingProcess[] = [];

export const recruitmentService = {
  async getJobPostings(filters?: any): Promise<JobPosting[]> {
    await new Promise(resolve => setTimeout(resolve, 1000));
    return mockJobPostings.filter(job => {
      if (filters?.status && job.status !== filters.status) return false;
      if (filters?.department && job.department !== filters.department) return false;
      return true;
    });
  },

  async createJobPosting(request: JobPostingRequest): Promise<JobPosting> {
    await new Promise(resolve => setTimeout(resolve, 1500));
    const newJob: JobPosting = {
      id: `job-${Date.now()}`,
      ...request,
      postedDate: new Date().toISOString().split('T')[0],
      status: 'Draft',
      applicationCount: 0,
      createdBy: 'current-user@company.com',
      updatedAt: new Date().toISOString()
    };
    mockJobPostings.push(newJob);
    return newJob;
  },

  async updateJobPosting(id: string, updates: Partial<JobPosting>): Promise<JobPosting> {
    await new Promise(resolve => setTimeout(resolve, 1000));
    const index = mockJobPostings.findIndex(job => job.id === id);
    if (index === -1) throw new Error('Job posting not found');
    
    mockJobPostings[index] = { ...mockJobPostings[index], ...updates, updatedAt: new Date().toISOString() };
    return mockJobPostings[index];
  },

  async deleteJobPosting(id: string): Promise<void> {
    await new Promise(resolve => setTimeout(resolve, 500));
    const index = mockJobPostings.findIndex(job => job.id === id);
    if (index === -1) throw new Error('Job posting not found');
    mockJobPostings.splice(index, 1);
  },

  async getCandidates(filters?: any): Promise<Candidate[]> {
    await new Promise(resolve => setTimeout(resolve, 1000));
    return mockCandidates.filter(candidate => {
      if (filters?.status && candidate.status !== filters.status) return false;
      if (filters?.source && candidate.source !== filters.source) return false;
      return true;
    });
  },

  async getCandidateById(id: string): Promise<Candidate> {
    await new Promise(resolve => setTimeout(resolve, 500));
    const candidate = mockCandidates.find(c => c.id === id);
    if (!candidate) throw new Error('Candidate not found');
    return candidate;
  },

  async createCandidate(request: CandidateRequest): Promise<Candidate> {
    await new Promise(resolve => setTimeout(resolve, 1500));
    const newCandidate: Candidate = {
      id: `candidate-${Date.now()}`,
      firstName: request.firstName,
      lastName: request.lastName,
      email: request.email,
      phone: request.phone,
      address: request.address,
      linkedInProfile: request.linkedInProfile,
      portfolioUrl: request.portfolioUrl,
      currentPosition: request.currentPosition,
      currentCompany: request.currentCompany,
      totalExperience: request.totalExperience,
      expectedSalary: request.expectedSalary,
      noticePeriod: request.noticePeriod,
      skills: request.skills,
      education: [],
      workExperience: [],
      appliedJobs: request.jobId ? [{
        id: `app-${Date.now()}`,
        jobId: request.jobId,
        jobTitle: mockJobPostings.find(j => j.id === request.jobId)?.title || '',
        appliedDate: new Date().toISOString().split('T')[0],
        status: 'Applied',
        stage: 'Application Received',
        interviewHistory: []
      }] : [],
      status: 'New',
      source: request.source as any,
      notes: '',
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString()
    };
    mockCandidates.push(newCandidate);
    return newCandidate;
  },

  async updateCandidate(id: string, updates: Partial<Candidate>): Promise<Candidate> {
    await new Promise(resolve => setTimeout(resolve, 1000));
    const index = mockCandidates.findIndex(candidate => candidate.id === id);
    if (index === -1) throw new Error('Candidate not found');
    
    mockCandidates[index] = { ...mockCandidates[index], ...updates, updatedAt: new Date().toISOString() };
    return mockCandidates[index];
  },

  async getInterviews(filters?: any): Promise<Interview[]> {
    await new Promise(resolve => setTimeout(resolve, 800));
    return mockInterviews.filter(interview => {
      if (filters?.status && interview.status !== filters.status) return false;
      if (filters?.type && interview.type !== filters.type) return false;
      return true;
    });
  },

  async scheduleInterview(request: InterviewRequest): Promise<Interview> {
    await new Promise(resolve => setTimeout(resolve, 1200));
    const newInterview: Interview = {
      id: `interview-${Date.now()}`,
      candidateId: request.candidateId,
      jobId: request.jobId,
      type: request.type,
      round: request.round,
      scheduledDate: request.scheduledDate,
      duration: request.duration,
      location: request.location,
      meetingLink: request.meetingLink,
      interviewers: request.interviewerIds.map((id, index) => ({
        id,
        name: `Interviewer ${index + 1}`,
        email: `interviewer${index + 1}@company.com`,
        role: 'Interviewer',
        department: 'Engineering',
        isLead: index === 0
      })),
      status: 'Scheduled',
      createdBy: 'current-user@company.com',
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString()
    };
    mockInterviews.push(newInterview);
    return newInterview;
  },

  async updateInterviewFeedback(interviewId: string, feedback: Omit<InterviewFeedback, 'id' | 'submittedAt'>): Promise<InterviewFeedback> {
    await new Promise(resolve => setTimeout(resolve, 1000));
    const newFeedback: InterviewFeedback = {
      ...feedback,
      id: `feedback-${Date.now()}`,
      submittedAt: new Date().toISOString()
    };
    
    const interview = mockInterviews.find(i => i.id === interviewId);
    if (interview) {
      interview.feedback = newFeedback;
      interview.status = 'Completed';
    }
    
    return newFeedback;
  },

  async getOfferLetters(): Promise<OfferLetter[]> {
    await new Promise(resolve => setTimeout(resolve, 800));
    return mockOfferLetters;
  },

  async createOfferLetter(offer: Omit<OfferLetter, 'id' | 'createdAt' | 'updatedAt'>): Promise<OfferLetter> {
    await new Promise(resolve => setTimeout(resolve, 1500));
    const newOffer: OfferLetter = {
      ...offer,
      id: `offer-${Date.now()}`,
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString()
    };
    mockOfferLetters.push(newOffer);
    return newOffer;
  },

  async updateOfferStatus(id: string, status: OfferLetter['status'], notes?: string): Promise<OfferLetter> {
    await new Promise(resolve => setTimeout(resolve, 800));
    const index = mockOfferLetters.findIndex(offer => offer.id === id);
    if (index === -1) throw new Error('Offer letter not found');
    
    mockOfferLetters[index] = { 
      ...mockOfferLetters[index], 
      status, 
      negotiationNotes: notes,
      responseDate: status !== 'Draft' && status !== 'Sent' ? new Date().toISOString() : undefined,
      updatedAt: new Date().toISOString() 
    };
    return mockOfferLetters[index];
  },

  async getOnboardingProcesses(): Promise<OnboardingProcess[]> {
    await new Promise(resolve => setTimeout(resolve, 1000));
    return mockOnboardingProcesses;
  },

  async createOnboardingProcess(request: OnboardingProcessRequest): Promise<OnboardingProcess> {
    await new Promise(resolve => setTimeout(resolve, 1500));
    
    const defaultTasks: OnboardingTask[] = [
      {
        id: 'task-001',
        title: 'Complete Employment Forms',
        description: 'Fill out all required employment documentation',
        category: 'Documentation',
        assignedTo: 'Employee',
        dueDate: new Date(Date.now() + 24 * 60 * 60 * 1000).toISOString(),
        priority: 'High',
        status: 'Pending'
      },
      {
        id: 'task-002',
        title: 'IT Equipment Setup',
        description: 'Assign laptop, phone, and access credentials',
        category: 'IT Setup',
        assignedTo: 'IT',
        dueDate: new Date(Date.now() + 48 * 60 * 60 * 1000).toISOString(),
        priority: 'High',
        status: 'Pending'
      },
      {
        id: 'task-003',
        title: 'Office Tour & Introduction',
        description: 'Introduce to team and show office facilities',
        category: 'Orientation',
        assignedTo: 'Manager',
        dueDate: new Date(Date.now() + 24 * 60 * 60 * 1000).toISOString(),
        priority: 'Medium',
        status: 'Pending'
      }
    ];
    
    const newProcess: OnboardingProcess = {
      id: `onboarding-${Date.now()}`,
      employeeId: request.employeeId,
      candidateId: request.candidateId,
      startDate: request.startDate,
      department: request.department,
      position: request.position,
      reportingManager: request.reportingManager,
      buddy: request.buddy,
      status: 'Not Started',
      progress: 0,
      tasks: [...defaultTasks, ...(request.customTasks?.map((task, index) => ({
        ...task,
        id: `custom-task-${index + 1}`
      })) || [])],
      welcomePackSent: false,
      workstationAssigned: false,
      accountsCreated: false,
      trainingScheduled: false,
      firstDayCompleted: false,
      firstWeekCompleted: false,
      probationReviewScheduled: false,
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString()
    };
    
    mockOnboardingProcesses.push(newProcess);
    return newProcess;
  },

  async updateOnboardingTask(processId: string, taskId: string, updates: Partial<OnboardingTask>): Promise<OnboardingProcess> {
    await new Promise(resolve => setTimeout(resolve, 800));
    const process = mockOnboardingProcesses.find(p => p.id === processId);
    if (!process) throw new Error('Onboarding process not found');
    
    const taskIndex = process.tasks.findIndex(t => t.id === taskId);
    if (taskIndex === -1) throw new Error('Task not found');
    
    process.tasks[taskIndex] = { ...process.tasks[taskIndex], ...updates };
    if (updates.status === 'Completed') {
      process.tasks[taskIndex].completedDate = new Date().toISOString();
    }
    
    const completedTasks = process.tasks.filter(t => t.status === 'Completed').length;
    process.progress = Math.round((completedTasks / process.tasks.length) * 100);
    process.updatedAt = new Date().toISOString();
    
    if (process.progress === 100) {
      process.status = 'Completed';
    } else if (process.progress > 0) {
      process.status = 'In Progress';
    }
    
    return process;
  },

  async getRecruitmentMetrics(): Promise<RecruitmentMetrics> {
    await new Promise(resolve => setTimeout(resolve, 1200));
    return {
      totalJobPostings: mockJobPostings.length,
      activeJobPostings: mockJobPostings.filter(j => j.status === 'Published').length,
      totalApplications: mockCandidates.length,
      newApplicationsToday: 3,
      interviewsScheduled: mockInterviews.filter(i => i.status === 'Scheduled').length,
      offersExtended: mockOfferLetters.filter(o => o.status === 'Sent').length,
      hiresCompleted: mockOnboardingProcesses.filter(p => p.status === 'Completed').length,
      averageTimeToHire: 28,
      averageCostPerHire: 45000,
      applicationToInterviewRate: 0.32,
      interviewToOfferRate: 0.68,
      offerAcceptanceRate: 0.85,
      sourceEffectiveness: {
        'LinkedIn': 35,
        'Website': 25,
        'Referral': 20,
        'Job Board': 15,
        'Agency': 5
      }
    };
  }
};