import { Employee, Department, Attendance, LeaveRequest, PayrollRecord, DashboardStats } from '../types';
import { format } from 'date-fns';

// Mock data generators for reports
const generateEmployeeReport = () => ({
  totalEmployees: 145,
  activeEmployees: 138,
  inactiveEmployees: 7,
  newHires: 12,
  terminations: 5,
  growthRate: 4.8,
  departments: [
    { name: 'Engineering', count: 45, percentage: 31.0 },
    { name: 'Sales', count: 32, percentage: 22.1 },
    { name: 'Marketing', count: 28, percentage: 19.3 },
    { name: 'HR', count: 15, percentage: 10.3 },
    { name: 'Finance', count: 25, percentage: 17.2 },
  ],
  ageDistribution: [
    { range: '20-30', count: 45 },
    { range: '31-40', count: 62 },
    { range: '41-50', count: 28 },
    { range: '51+', count: 10 },
  ],
  tenureDistribution: [
    { range: '0-1 years', count: 35 },
    { range: '1-3 years', count: 58 },
    { range: '3-5 years', count: 32 },
    { range: '5+ years', count: 20 },
  ]
});

const generateAttendanceReport = () => ({
  overallRate: 92.5,
  dailyAverage: 134,
  lateArrivals: 45,
  earlyDepartures: 23,
  overtime: {
    totalHours: 245,
    totalEmployees: 67,
    averagePerEmployee: 3.7
  },
  departmentWise: [
    { department: 'Engineering', rate: 94.2, avgHours: 8.2 },
    { department: 'Sales', rate: 89.8, avgHours: 7.9 },
    { department: 'Marketing', rate: 91.5, avgHours: 8.1 },
    { department: 'HR', rate: 96.7, avgHours: 8.0 },
    { department: 'Finance', rate: 93.2, avgHours: 8.3 },
  ],
  trends: [
    { month: 'Jan', rate: 89.5 },
    { month: 'Feb', rate: 91.2 },
    { month: 'Mar', rate: 93.8 },
    { month: 'Apr', rate: 92.1 },
    { month: 'May', rate: 90.7 },
    { month: 'Jun', rate: 92.5 },
  ]
});

const generateLeaveReport = () => ({
  totalRequests: 156,
  approved: 142,
  rejected: 8,
  pending: 6,
  approvalRate: 91.0,
  totalDaysTaken: 1247,
  averageDaysPerEmployee: 8.6,
  byType: {
    sick: { count: 45, days: 67 },
    vacation: { count: 78, days: 890 },
    personal: { count: 23, days: 45 },
    maternity: { count: 8, days: 720 },
    paternity: { count: 2, days: 24 }
  },
  monthlyTrends: [
    { month: 'Jan', requests: 18, days: 156 },
    { month: 'Feb', requests: 22, days: 189 },
    { month: 'Mar', requests: 28, days: 234 },
    { month: 'Apr', requests: 25, days: 201 },
    { month: 'May', requests: 32, days: 289 },
    { month: 'Jun', requests: 31, days: 278 },
  ],
  departmentWise: [
    { department: 'Engineering', requests: 52, days: 456 },
    { department: 'Sales', requests: 38, days: 298 },
    { department: 'Marketing', requests: 34, days: 267 },
    { department: 'HR', requests: 16, days: 123 },
    { department: 'Finance', requests: 26, days: 203 },
  ]
});

const generatePayrollReport = () => ({
  totalAmount: 6750000,
  averageSalary: 46552,
  medianSalary: 42000,
  totalEmployees: 145,
  salaryDistribution: [
    { range: '30K-50K', count: 65, percentage: 44.8 },
    { range: '50K-70K', count: 45, percentage: 31.0 },
    { range: '70K-90K', count: 25, percentage: 17.2 },
    { range: '90K+', count: 10, percentage: 6.9 },
  ],
  departmentWise: [
    { department: 'Engineering', totalAmount: 2250000, avgSalary: 50000, employees: 45 },
    { department: 'Sales', totalAmount: 1440000, avgSalary: 45000, employees: 32 },
    { department: 'Marketing', totalAmount: 1260000, avgSalary: 45000, employees: 28 },
    { department: 'HR', totalAmount: 675000, avgSalary: 45000, employees: 15 },
    { department: 'Finance', totalAmount: 1125000, avgSalary: 45000, employees: 25 },
  ],
  monthlyTrends: [
    { month: 'Jan', amount: 6200000 },
    { month: 'Feb', amount: 6350000 },
    { month: 'Mar', amount: 6500000 },
    { month: 'Apr', amount: 6650000 },
    { month: 'May', amount: 6700000 },
    { month: 'Jun', amount: 6750000 },
  ],
  costs: {
    basicSalaries: 5400000,
    allowances: 675000,
    overtime: 337500,
    deductions: 202500,
    netPayable: 6210000,
  }
});

export const reportsService = {
  // Generate comprehensive employee report
  generateEmployeeReport: async (
    startDate?: string,
    endDate?: string,
    departmentFilter?: string
  ): Promise<any> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const report = generateEmployeeReport();
        resolve({
          ...report,
          generatedAt: format(new Date(), 'yyyy-MM-dd HH:mm:ss'),
          period: startDate && endDate ? `${startDate} to ${endDate}` : 'All time',
          filters: { department: departmentFilter || 'All' },
        });
      }, 1000);
    });
  },

  // Generate attendance report
  generateAttendanceReport: async (
    startDate?: string,
    endDate?: string,
    employeeId?: string,
    departmentFilter?: string
  ): Promise<any> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const report = generateAttendanceReport();
        resolve({
          ...report,
          generatedAt: format(new Date(), 'yyyy-MM-dd HH:mm:ss'),
          period: startDate && endDate ? `${startDate} to ${endDate}` : 'Last 6 months',
          filters: { 
            employee: employeeId || 'All',
            department: departmentFilter || 'All'
          },
        });
      }, 1000);
    });
  },

  // Generate leave report
  generateLeaveReport: async (
    startDate?: string,
    endDate?: string,
    leaveType?: string,
    departmentFilter?: string
  ): Promise<any> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const report = generateLeaveReport();
        resolve({
          ...report,
          generatedAt: format(new Date(), 'yyyy-MM-dd HH:mm:ss'),
          period: startDate && endDate ? `${startDate} to ${endDate}` : 'Last 6 months',
          filters: { 
            leaveType: leaveType || 'All',
            department: departmentFilter || 'All'
          },
        });
      }, 1000);
    });
  },

  // Generate payroll report
  generatePayrollReport: async (
    startMonth?: string,
    startYear?: number,
    endMonth?: string,
    endYear?: number,
    departmentFilter?: string
  ): Promise<any> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const report = generatePayrollReport();
        resolve({
          ...report,
          generatedAt: format(new Date(), 'yyyy-MM-dd HH:mm:ss'),
          period: startMonth && startYear && endMonth && endYear 
            ? `${startMonth} ${startYear} to ${endMonth} ${endYear}` 
            : 'Last 6 months',
          filters: { department: departmentFilter || 'All' },
        });
      }, 1000);
    });
  },

  // Generate department performance report
  generateDepartmentReport: async (): Promise<any> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const departmentData = [
          {
            name: 'Engineering',
            employees: 45,
            budget: 2500000,
            utilization: 95.2,
            avgSalary: 55556,
            attendanceRate: 94.2,
            leaveUtilization: 78.5,
            performance: 92.8,
            growthRate: 8.2
          },
          {
            name: 'Sales',
            employees: 32,
            budget: 1800000,
            utilization: 87.3,
            avgSalary: 56250,
            attendanceRate: 89.8,
            leaveUtilization: 82.1,
            performance: 88.5,
            growthRate: 12.5
          },
          {
            name: 'Marketing',
            employees: 28,
            budget: 1200000,
            utilization: 91.8,
            avgSalary: 42857,
            attendanceRate: 91.5,
            leaveUtilization: 75.3,
            performance: 90.2,
            growthRate: 5.7
          },
          {
            name: 'HR',
            employees: 15,
            budget: 800000,
            utilization: 98.1,
            avgSalary: 53333,
            attendanceRate: 96.7,
            leaveUtilization: 68.9,
            performance: 94.5,
            growthRate: 3.4
          },
          {
            name: 'Finance',
            employees: 25,
            budget: 1000000,
            utilization: 93.5,
            avgSalary: 40000,
            attendanceRate: 93.2,
            leaveUtilization: 71.2,
            performance: 91.8,
            growthRate: 6.1
          }
        ];

        resolve({
          departments: departmentData,
          summary: {
            totalDepartments: departmentData.length,
            totalEmployees: departmentData.reduce((sum, d) => sum + d.employees, 0),
            totalBudget: departmentData.reduce((sum, d) => sum + d.budget, 0),
            avgPerformance: departmentData.reduce((sum, d) => sum + d.performance, 0) / departmentData.length,
            avgAttendanceRate: departmentData.reduce((sum, d) => sum + d.attendanceRate, 0) / departmentData.length,
          },
          generatedAt: format(new Date(), 'yyyy-MM-dd HH:mm:ss'),
        });
      }, 1000);
    });
  },

  // Export report data
  exportReport: async (
    reportType: 'employee' | 'attendance' | 'leave' | 'payroll' | 'department',
    exportFormat: 'csv' | 'excel' | 'pdf',
    data: any,
    filename?: string
  ): Promise<{
    success: boolean;
    downloadUrl?: string;
    filename: string;
    format: string;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const timestamp = format(new Date(), 'yyyyMMdd_HHmmss');
        const generatedFilename = filename || `${reportType}_report_${timestamp}.${exportFormat}`;
        
        // Mock export process - in real implementation, this would generate actual files
        resolve({
          success: true,
          downloadUrl: `/api/downloads/${generatedFilename}`,
          filename: generatedFilename,
          format: exportFormat,
        });
      }, 2000);
    });
  },

  // Get dashboard statistics
  getDashboardStats: async (): Promise<DashboardStats> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const stats: DashboardStats = {
          totalEmployees: 145,
          activeEmployees: 138,
          totalDepartments: 8,
          pendingLeaves: 12,
          todayAttendance: 134,
          monthlyPayroll: 6750000,
        };
        resolve(stats);
      }, 500);
    });
  },

  // Generate custom report
  generateCustomReport: async (config: {
    reportName: string;
    dataSource: string[];
    dateRange: { start: string; end: string };
    filters: { [key: string]: any };
    groupBy?: string;
    sortBy?: string;
    includeCharts: boolean;
  }): Promise<{
    reportId: string;
    name: string;
    data: any;
    charts?: any[];
    summary: any;
    generatedAt: string;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        // Mock custom report generation
        const reportData = {
          reportId: Date.now().toString(),
          name: config.reportName,
          data: {
            records: [], // Would contain actual filtered and processed data
            totalRecords: 0,
            period: `${config.dateRange.start} to ${config.dateRange.end}`,
          },
          charts: config.includeCharts ? [
            { type: 'bar', title: 'Monthly Trends', data: [] },
            { type: 'pie', title: 'Distribution', data: [] },
          ] : undefined,
          summary: {
            totalRecords: 0,
            keyMetrics: {},
          },
          generatedAt: format(new Date(), 'yyyy-MM-dd HH:mm:ss'),
        };
        
        resolve(reportData);
      }, 1500);
    });
  },

  // Get report templates
  getReportTemplates: async (): Promise<Array<{
    id: string;
    name: string;
    description: string;
    category: string;
    fields: string[];
    defaultFilters: any;
  }>> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const templates = [
          {
            id: 'employee_master',
            name: 'Employee Master Report',
            description: 'Complete employee information with personal and job details',
            category: 'Employee',
            fields: ['name', 'employeeId', 'department', 'position', 'salary', 'hireDate', 'status'],
            defaultFilters: { status: 'active' }
          },
          {
            id: 'monthly_attendance',
            name: 'Monthly Attendance Report',
            description: 'Monthly attendance summary with working hours and overtime',
            category: 'Attendance',
            fields: ['employee', 'presentDays', 'absentDays', 'workingHours', 'overtime'],
            defaultFilters: { month: format(new Date(), 'MMMM'), year: new Date().getFullYear() }
          },
          {
            id: 'leave_summary',
            name: 'Leave Summary Report',
            description: 'Leave requests summary by type and status',
            category: 'Leave',
            fields: ['employee', 'leaveType', 'startDate', 'endDate', 'days', 'status'],
            defaultFilters: { status: 'all' }
          },
          {
            id: 'payroll_register',
            name: 'Payroll Register',
            description: 'Detailed payroll information with salary breakdown',
            category: 'Payroll',
            fields: ['employee', 'basicSalary', 'allowances', 'deductions', 'overtime', 'netSalary'],
            defaultFilters: { month: format(new Date(), 'MMMM') }
          },
          {
            id: 'department_analytics',
            name: 'Department Analytics',
            description: 'Department-wise performance and cost analysis',
            category: 'Department',
            fields: ['department', 'employees', 'budget', 'avgSalary', 'performance'],
            defaultFilters: {}
          }
        ];
        resolve(templates);
      }, 300);
    });
  },

  // Save custom report
  saveCustomReport: async (reportData: any): Promise<{
    success: boolean;
    reportId: string;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        // Mock save operation
        resolve({
          success: true,
          reportId: Date.now().toString(),
        });
      }, 500);
    });
  },

  // Get saved reports
  getSavedReports: async (): Promise<Array<{
    id: string;
    name: string;
    type: string;
    createdAt: string;
    lastModified: string;
    size: string;
  }>> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const savedReports = [
          {
            id: '1',
            name: 'Q2 Employee Report',
            type: 'employee',
            createdAt: '2023-06-15',
            lastModified: '2023-06-20',
            size: '2.5 MB'
          },
          {
            id: '2',
            name: 'June Attendance Analysis',
            type: 'attendance',
            createdAt: '2023-06-30',
            lastModified: '2023-06-30',
            size: '1.8 MB'
          },
          {
            id: '3',
            name: 'Annual Leave Summary',
            type: 'leave',
            createdAt: '2023-01-01',
            lastModified: '2023-06-25',
            size: '3.2 MB'
          }
        ];
        resolve(savedReports);
      }, 300);
    });
  },

  // Generate performance metrics
  generatePerformanceMetrics: async (employeeId?: string, departmentFilter?: string): Promise<{
    overall: {
      averagePerformance: number;
      topPerformers: Array<{
        employeeId: string;
        name: string;
        department: string;
        score: number;
      }>;
      improvementNeeded: Array<{
        employeeId: string;
        name: string;
        department: string;
        score: number;
      }>;
    };
    departmentWise: Array<{
      department: string;
      averageScore: number;
      employeeCount: number;
      topPerformer: string;
    }>;
    trends: Array<{
      month: string;
      averageScore: number;
      improvement: number;
    }>;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const metrics = {
          overall: {
            averagePerformance: 87.5,
            topPerformers: [
              { employeeId: '1', name: 'John Doe', department: 'Engineering', score: 98.5 },
              { employeeId: '2', name: 'Sarah Johnson', department: 'Sales', score: 96.2 },
              { employeeId: '3', name: 'Michael Brown', department: 'Marketing', score: 94.8 },
            ],
            improvementNeeded: [
              { employeeId: '10', name: 'Alex Wilson', department: 'Sales', score: 72.3 },
              { employeeId: '11', name: 'Lisa Chen', department: 'Marketing', score: 74.1 },
            ]
          },
          departmentWise: [
            { department: 'Engineering', averageScore: 91.2, employeeCount: 45, topPerformer: 'John Doe' },
            { department: 'Sales', averageScore: 85.8, employeeCount: 32, topPerformer: 'Sarah Johnson' },
            { department: 'Marketing', averageScore: 87.4, employeeCount: 28, topPerformer: 'Michael Brown' },
            { department: 'HR', averageScore: 89.1, employeeCount: 15, topPerformer: 'Emily Davis' },
            { department: 'Finance', averageScore: 88.7, employeeCount: 25, topPerformer: 'David Wilson' },
          ],
          trends: [
            { month: 'Jan', averageScore: 85.2, improvement: 2.1 },
            { month: 'Feb', averageScore: 86.1, improvement: 0.9 },
            { month: 'Mar', averageScore: 87.3, improvement: 1.2 },
            { month: 'Apr', averageScore: 86.8, improvement: -0.5 },
            { month: 'May', averageScore: 88.1, improvement: 1.3 },
            { month: 'Jun', averageScore: 87.5, improvement: -0.6 },
          ]
        };
        
        resolve(metrics);
      }, 1000);
    });
  },

  // Generate executive summary
  generateExecutiveSummary: async (): Promise<{
    summary: {
      totalEmployees: number;
      payrollCosts: number;
      attendanceRate: number;
      leaveUtilization: number;
      departmentCount: number;
    };
    keyInsights: string[];
    recommendations: string[];
    riskFactors: string[];
    generatedAt: string;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const summary = {
          summary: {
            totalEmployees: 145,
            payrollCosts: 6750000,
            attendanceRate: 92.5,
            leaveUtilization: 76.8,
            departmentCount: 5,
          },
          keyInsights: [
            'Employee count increased by 12% this quarter',
            'Engineering department shows highest attendance rate at 94.2%',
            'Average leave utilization is within industry standards',
            'Payroll costs increased by 8.5% due to new hires and promotions'
          ],
          recommendations: [
            'Consider implementing flexible working hours to improve attendance',
            'Review leave policies for better work-life balance',
            'Invest in employee development programs for retention',
            'Optimize department budgets based on performance metrics'
          ],
          riskFactors: [
            'Higher than average turnover in Sales department',
            'Increasing overtime costs in Engineering',
            'Pending leave approvals may affect project deadlines'
          ],
          generatedAt: format(new Date(), 'yyyy-MM-dd HH:mm:ss'),
        };
        
        resolve(summary);
      }, 1500);
    });
  }
};

// Helper function to get month name from number
function getMonthName(monthNumber: number): string {
  const months = [
    'January', 'February', 'March', 'April', 'May', 'June',
    'July', 'August', 'September', 'October', 'November', 'December'
  ];
  return months[monthNumber - 1] || 'January';
}