// API-specific types for HTTP requests and responses

// HTTP Method Types
export type HttpMethod = 'GET' | 'POST' | 'PUT' | 'PATCH' | 'DELETE';

// Request Configuration
export interface ApiRequestConfig {
  method: HttpMethod;
  url: string;
  headers?: Record<string, string>;
  params?: Record<string, any>;
  data?: any;
  timeout?: number;
}

// Response Structure
export interface ApiResponse<T = any> {
  success: boolean;
  data: T;
  message?: string;
  errors?: ApiError[];
  metadata?: ResponseMetadata;
}

export interface ResponseMetadata {
  timestamp: string;
  requestId: string;
  version: string;
  pagination?: PaginationMetadata;
  cache?: CacheMetadata;
}

export interface PaginationMetadata {
  page: number;
  limit: number;
  total: number;
  totalPages: number;
  hasNext: boolean;
  hasPrevious: boolean;
  nextPage?: number;
  previousPage?: number;
}

export interface CacheMetadata {
  cached: boolean;
  cacheKey?: string;
  cachedAt?: string;
  expiresAt?: string;
  ttl?: number;
}

// Error Types
export interface ApiError {
  code: string;
  message: string;
  field?: string;
  details?: any;
}

export interface ValidationError extends ApiError {
  field: string;
  rejectedValue?: any;
}

// Authentication
export interface AuthRequest {
  email: string;
  password: string;
  rememberMe?: boolean;
}

export interface AuthResponse {
  user: {
    id: string;
    email: string;
    name: string;
    role: string;
  };
  tokens: {
    accessToken: string;
    refreshToken: string;
    expiresAt: string;
  };
  permissions: string[];
}

export interface RefreshTokenRequest {
  refreshToken: string;
}

export interface RefreshTokenResponse {
  accessToken: string;
  expiresAt: string;
}

// File Upload
export interface FileUploadRequest {
  file: File;
  uploadType: 'avatar' | 'document' | 'report' | 'import';
  metadata?: Record<string, any>;
}

export interface FileUploadResponse {
  fileId: string;
  filename: string;
  url: string;
  size: number;
  mimeType: string;
  uploadedAt: string;
}

// Bulk Operations
export interface BulkRequest<T> {
  operation: 'create' | 'update' | 'delete';
  data: T[];
  options?: {
    stopOnError?: boolean;
    validateOnly?: boolean;
    batchSize?: number;
  };
}

export interface BulkResponse {
  totalRequests: number;
  successCount: number;
  errorCount: number;
  results: Array<{
    index: number;
    success: boolean;
    data?: any;
    error?: ApiError;
  }>;
  summary: {
    processedAt: string;
    duration: number; // milliseconds
    warnings?: string[];
  };
}

// Search & Filtering
export interface SearchRequest {
  query: string;
  filters?: Record<string, any>;
  sort?: {
    field: string;
    order: 'asc' | 'desc';
  }[];
  pagination?: {
    page: number;
    limit: number;
  };
  highlight?: boolean;
}

export interface SearchResponse<T> {
  results: T[];
  total: number;
  took: number; // milliseconds
  suggestions?: string[];
  facets?: Record<string, Array<{
    value: string;
    count: number;
  }>>;
  pagination: PaginationMetadata;
}

// Export Operations
export interface ExportRequest {
  type: 'csv' | 'excel' | 'pdf' | 'json';
  data?: any;
  template?: string;
  filters?: Record<string, any>;
  options?: {
    filename?: string;
    includeHeaders?: boolean;
    dateFormat?: string;
  };
}

export interface ExportResponse {
  exportId: string;
  status: 'pending' | 'processing' | 'completed' | 'failed';
  downloadUrl?: string;
  filename: string;
  size?: number;
  createdAt: string;
  expiresAt: string;
  error?: string;
}

// Import Operations
export interface ImportRequest {
  fileId: string;
  type: 'employees' | 'attendance' | 'leaves' | 'payroll';
  options?: {
    skipDuplicates?: boolean;
    validateOnly?: boolean;
    mapping?: Record<string, string>;
  };
}

export interface ImportResponse {
  importId: string;
  status: 'pending' | 'processing' | 'completed' | 'failed';
  totalRecords: number;
  processedRecords: number;
  successCount: number;
  errorCount: number;
  warnings: string[];
  errors: Array<{
    row: number;
    field?: string;
    message: string;
  }>;
  startedAt: string;
  completedAt?: string;
}

// Real-time Events
export interface WebSocketMessage {
  type: 'notification' | 'update' | 'alert' | 'system';
  event: string;
  data: any;
  timestamp: string;
  userId?: string;
}

export interface NotificationEvent {
  id: string;
  type: 'info' | 'success' | 'warning' | 'error';
  title: string;
  message: string;
  timestamp: string;
  actionUrl?: string;
  metadata?: Record<string, any>;
}

// API Endpoints
export interface ApiEndpoints {
  // Authentication
  auth: {
    login: string;
    logout: string;
    refresh: string;
    resetPassword: string;
  };
  
  // Employees
  employees: {
    list: string;
    get: string;
    create: string;
    update: string;
    delete: string;
    bulk: string;
    search: string;
    export: string;
    import: string;
  };
  
  // Departments
  departments: {
    list: string;
    get: string;
    create: string;
    update: string;
    delete: string;
    employees: string;
  };
  
  // Attendance
  attendance: {
    list: string;
    get: string;
    create: string;
    update: string;
    delete: string;
    checkIn: string;
    checkOut: string;
    timesheet: string;
    reports: string;
  };
  
  // Leave Management
  leaves: {
    list: string;
    get: string;
    create: string;
    update: string;
    delete: string;
    approve: string;
    reject: string;
    balance: string;
    calendar: string;
  };
  
  // Payroll
  payroll: {
    list: string;
    get: string;
    create: string;
    update: string;
    delete: string;
    process: string;
    payslip: string;
    batch: string;
  };
  
  // Reports
  reports: {
    generate: string;
    export: string;
    templates: string;
    analytics: string;
  };
  
  // Notifications
  notifications: {
    list: string;
    get: string;
    create: string;
    markRead: string;
    markAllRead: string;
  };
  
  // Dashboard
  dashboard: {
    stats: string;
    metrics: string;
    activity: string;
    widgets: string;
  };
  
  // File Operations
  files: {
    upload: string;
    download: string;
    delete: string;
  };
}

// Request Builders
export interface RequestBuilder {
  get<T>(url: string, params?: any): Promise<ApiResponse<T>>;
  post<T>(url: string, data?: any): Promise<ApiResponse<T>>;
  put<T>(url: string, data?: any): Promise<ApiResponse<T>>;
  patch<T>(url: string, data?: any): Promise<ApiResponse<T>>;
  delete<T>(url: string): Promise<ApiResponse<T>>;
}

// API Client Configuration
export interface ApiClientConfig {
  baseURL: string;
  timeout: number;
  retries: number;
  retryDelay: number;
  headers: Record<string, string>;
  interceptors?: {
    request?: (config: ApiRequestConfig) => ApiRequestConfig;
    response?: <T>(response: ApiResponse<T>) => ApiResponse<T>;
    error?: (error: any) => any;
  };
}

// Status Codes
export enum HttpStatusCode {
  OK = 200,
  Created = 201,
  NoContent = 204,
  BadRequest = 400,
  Unauthorized = 401,
  Forbidden = 403,
  NotFound = 404,
  Conflict = 409,
  UnprocessableEntity = 422,
  TooManyRequests = 429,
  InternalServerError = 500,
  BadGateway = 502,
  ServiceUnavailable = 503,
  GatewayTimeout = 504,
}

// Rate Limiting
export interface RateLimitInfo {
  limit: number;
  remaining: number;
  reset: number; // Unix timestamp
  retryAfter?: number; // seconds
}

// API Health
export interface HealthCheckResponse {
  status: 'healthy' | 'degraded' | 'unhealthy';
  timestamp: string;
  uptime: number;
  version: string;
  services: Array<{
    name: string;
    status: 'up' | 'down';
    responseTime?: number;
    lastCheck: string;
  }>;
}

// Webhook
export interface WebhookPayload {
  id: string;
  event: string;
  data: any;
  timestamp: string;
  signature: string;
  version: string;
}

export interface WebhookConfiguration {
  url: string;
  events: string[];
  secret: string;
  isActive: boolean;
  retryPolicy: {
    maxRetries: number;
    retryInterval: number;
  };
}