import { Attendance } from '../index';

// Attendance service specific types
export interface AttendanceFilter {
  employeeId?: string;
  department?: string;
  dateRange?: {
    startDate: string;
    endDate: string;
  };
  status?: Attendance['status'];
  workingHoursRange?: {
    min: number;
    max: number;
  };
  hasOvertime?: boolean;
}

export interface AttendanceSortOptions {
  field: 'date' | 'employeeId' | 'checkIn' | 'checkOut' | 'workingHours' | 'overtime';
  order: 'asc' | 'desc';
}

export interface AttendancePaginationOptions {
  page: number;
  limit: number;
  sort?: AttendanceSortOptions;
}

export interface AttendanceListResponse {
  attendance: Attendance[];
  total: number;
  page: number;
  limit: number;
  totalPages: number;
}

export interface AttendanceCreateRequest extends Omit<Attendance, 'id'> {}

export interface AttendanceUpdateRequest extends Partial<Omit<Attendance, 'id'>> {}

export interface AttendanceBulkCreateRequest {
  attendance: AttendanceCreateRequest[];
  validateDuplicates?: boolean;
}

export interface AttendanceStats {
  totalRecords: number;
  presentCount: number;
  absentCount: number;
  lateCount: number;
  halfDayCount: number;
  attendanceRate: number;
  avgWorkingHours: number;
  totalOvertimeHours: number;
  punctualityRate: number;
}

export interface DailyAttendanceReport {
  date: string;
  totalEmployees: number;
  present: number;
  absent: number;
  late: number;
  halfDay: number;
  attendanceRate: number;
  departmentBreakdown: Array<{
    department: string;
    present: number;
    total: number;
    rate: number;
  }>;
}

export interface WeeklyAttendanceReport {
  weekStartDate: string;
  weekEndDate: string;
  dailyReports: DailyAttendanceReport[];
  weeklyStats: AttendanceStats;
  trends: {
    attendanceRateTrend: number; // positive/negative change
    punctualityTrend: number;
    overtimeTrend: number;
  };
}

export interface MonthlyAttendanceReport {
  month: string;
  year: number;
  weeklyReports: WeeklyAttendanceReport[];
  monthlyStats: AttendanceStats;
  employeeAttendance: Array<{
    employeeId: string;
    employeeName: string;
    totalDays: number;
    presentDays: number;
    absentDays: number;
    lateDays: number;
    attendanceRate: number;
    totalWorkingHours: number;
    overtimeHours: number;
  }>;
}

export interface AttendanceTimesheet {
  employeeId: string;
  employeeName: string;
  period: {
    startDate: string;
    endDate: string;
  };
  records: Attendance[];
  summary: {
    totalDays: number;
    workingDays: number;
    presentDays: number;
    absentDays: number;
    lateDays: number;
    totalHours: number;
    regularHours: number;
    overtimeHours: number;
    attendanceRate: number;
  };
}

export interface AttendancePattern {
  employeeId: string;
  patterns: {
    frequentLateArrival: boolean;
    frequentEarlyDeparture: boolean;
    inconsistentSchedule: boolean;
    weekendWork: boolean;
    excessiveOvertime: boolean;
  };
  recommendations: string[];
}

export interface AttendanceAlert {
  id: string;
  type: 'absence' | 'late_arrival' | 'missing_checkout' | 'excessive_overtime';
  employeeId: string;
  date: string;
  message: string;
  severity: 'low' | 'medium' | 'high';
  isResolved: boolean;
}

export interface AttendanceGeolocation {
  latitude: number;
  longitude: number;
  address?: string;
  accuracy?: number;
}

export interface AttendanceCheckIn {
  employeeId: string;
  timestamp: string;
  location?: AttendanceGeolocation;
  deviceInfo?: {
    deviceId: string;
    platform: string;
    appVersion: string;
  };
}

export interface AttendanceCheckOut extends AttendanceCheckIn {}

export interface AttendanceValidationRule {
  id: string;
  name: string;
  description: string;
  rule: {
    minWorkingHours?: number;
    maxWorkingHours?: number;
    requiredCheckIn?: boolean;
    requiredCheckOut?: boolean;
    allowedLocations?: AttendanceGeolocation[];
    restrictedDays?: string[];
  };
  isActive: boolean;
}