import { BaseFilter, PaginatedResponse } from './index';

// HR Policy Types
export interface HRPolicy {
  id: string;
  title: string;
  description: string;
  category: 'attendance' | 'leave' | 'code_of_conduct' | 'disciplinary' | 'benefits' | 'safety' | 'other';
  version: string;
  content: string; // Rich text content
  effectiveDate: string;
  expiryDate?: string;
  status: 'draft' | 'active' | 'archived' | 'expired';
  attachments?: PolicyAttachment[];
  approvedBy?: string;
  approvedDate?: string;
  createdBy: string;
  createdAt: string;
  updatedAt: string;
  tags?: string[];
  acknowledgmentRequired: boolean;
  acknowledgmentCount?: number;
  applicableTo: 'all' | 'department' | 'role' | 'specific';
  applicableIds?: string[]; // Department/Role/Employee IDs
}

export interface PolicyAttachment {
  id: string;
  fileName: string;
  fileUrl: string;
  fileSize: number;
  uploadedAt: string;
  uploadedBy: string;
}

export interface PolicyAcknowledgment {
  id: string;
  policyId: string;
  employeeId: string;
  acknowledgedAt: string;
  version: string;
  ipAddress?: string;
}

// Legal Compliance Types
export interface ComplianceRule {
  id: string;
  name: string;
  type: 'pf' | 'esi' | 'gratuity' | 'tds' | 'pt' | 'labour_law' | 'other';
  description: string;
  applicableFrom: string;
  applicableTo?: string;
  isActive: boolean;
  configuration: ComplianceConfiguration;
  lastUpdated: string;
  updatedBy: string;
}

export interface ComplianceConfiguration {
  // PF Configuration
  pfEmployeeRate?: number; // Percentage
  pfEmployerRate?: number; // Percentage
  pfWageLimit?: number; // Monthly wage limit
  pfEpsRate?: number; // EPS rate
  pfEdliRate?: number; // EDLI rate
  
  // ESI Configuration
  esiEmployeeRate?: number; // Percentage
  esiEmployerRate?: number; // Percentage
  esiWageLimit?: number; // Monthly wage limit
  
  // Gratuity Configuration
  gratuityWageLimit?: number; // Monthly wage limit
  gratuityYearsRequired?: number; // Minimum years of service
  gratuityRate?: number; // Days per year
  gratuityMaxLimit?: number; // Maximum gratuity amount
  
  // TDS Configuration
  tdsRates?: TDSRate[];
  
  // Professional Tax Configuration
  ptSlabs?: PTSlab[];
  
  // Other configurations
  customRules?: CustomRule[];
}

export interface TDSRate {
  incomeFrom: number;
  incomeTo?: number;
  rate: number;
  description: string;
}

export interface PTSlab {
  salaryFrom: number;
  salaryTo?: number;
  amount: number;
  state: string;
}

export interface CustomRule {
  name: string;
  formula: string;
  conditions: string;
}

// Compliance Tracking
export interface ComplianceRecord {
  id: string;
  employeeId: string;
  employeeName: string;
  month: string;
  year: number;
  salary: number;
  
  // PF Details
  pfEmployee: number;
  pfEmployer: number;
  pfEps: number;
  pfEdli: number;
  pfWages: number;
  pfUAN?: string;
  
  // ESI Details
  esiEmployee: number;
  esiEmployer: number;
  esiWages: number;
  esiNumber?: string;
  
  // Gratuity Details
  gratuityProvision: number;
  gratuityWages: number;
  
  // TDS Details
  tdsAmount: number;
  tdsSection: string;
  
  // PT Details
  ptAmount: number;
  ptState: string;
  
  status: 'calculated' | 'verified' | 'submitted' | 'paid';
  calculatedAt: string;
  calculatedBy: string;
  verifiedAt?: string;
  verifiedBy?: string;
  submittedAt?: string;
  paidAt?: string;
  
  notes?: string;
  errors?: string[];
}

// Compliance Reports
export interface ComplianceReport {
  id: string;
  name: string;
  type: 'pf_monthly' | 'esi_monthly' | 'tds_quarterly' | 'gratuity_annual' | 'pt_monthly' | 'custom';
  period: {
    from: string;
    to: string;
  };
  generatedAt: string;
  generatedBy: string;
  status: 'generating' | 'completed' | 'failed';
  fileUrl?: string;
  summary: ComplianceReportSummary;
  details?: any[];
}

export interface ComplianceReportSummary {
  totalEmployees: number;
  totalWages: number;
  totalPFEmployee: number;
  totalPFEmployer: number;
  totalESIEmployee: number;
  totalESIEmployer: number;
  totalTDS: number;
  totalPT: number;
  totalGratuity: number;
  errors: number;
  warnings: number;
}

// Audit Trail
export interface ComplianceAudit {
  id: string;
  action: 'created' | 'updated' | 'deleted' | 'calculated' | 'verified' | 'submitted';
  entityType: 'policy' | 'compliance_rule' | 'compliance_record' | 'report';
  entityId: string;
  userId: string;
  userName: string;
  timestamp: string;
  oldValue?: any;
  newValue?: any;
  ipAddress?: string;
}

// Service Request/Response Types
export interface PolicyFilter extends BaseFilter {
  category?: HRPolicy['category'];
  status?: HRPolicy['status'];
  approvedBy?: string;
  effectiveDateFrom?: string;
  effectiveDateTo?: string;
  tags?: string[];
}

export interface PolicyListResponse extends PaginatedResponse<HRPolicy> {
  categories: Array<{
    category: HRPolicy['category'];
    count: number;
  }>;
}

export interface ComplianceFilter extends BaseFilter {
  type?: ComplianceRule['type'];
  month?: string;
  year?: number;
  employeeId?: string;
  department?: string;
  status?: ComplianceRecord['status'];
}

export interface ComplianceListResponse extends PaginatedResponse<ComplianceRecord> {
  summary: ComplianceReportSummary;
}

export interface PolicyCreateRequest {
  title: string;
  description: string;
  category: HRPolicy['category'];
  content: string;
  effectiveDate: string;
  expiryDate?: string;
  acknowledgmentRequired: boolean;
  applicableTo: HRPolicy['applicableTo'];
  applicableIds?: string[];
  tags?: string[];
  attachments?: File[];
}

export interface PolicyUpdateRequest extends Partial<PolicyCreateRequest> {
  status?: HRPolicy['status'];
  version?: string;
}

export interface ComplianceRuleCreateRequest {
  name: string;
  type: ComplianceRule['type'];
  description: string;
  applicableFrom: string;
  applicableTo?: string;
  configuration: ComplianceConfiguration;
}

export interface ComplianceRuleUpdateRequest extends Partial<ComplianceRuleCreateRequest> {
  isActive?: boolean;
}

export interface ComplianceCalculationRequest {
  employeeIds?: string[];
  month: string;
  year: number;
  forceRecalculate?: boolean;
}

export interface BulkComplianceOperationRequest {
  operation: 'calculate' | 'verify' | 'submit' | 'export';
  filters: ComplianceFilter;
  params?: {
    format?: 'excel' | 'pdf' | 'csv';
    includeSummary?: boolean;
  };
}

export interface ComplianceStats {
  totalPolicies: number;
  activePolicies: number;
  draftPolicies: number;
  expiredPolicies: number;
  pendingAcknowledgments: number;
  
  totalComplianceRecords: number;
  calculatedRecords: number;
  verifiedRecords: number;
  submittedRecords: number;
  errorRecords: number;
  
  monthlyTotals: {
    wages: number;
    pfEmployee: number;
    pfEmployer: number;
    esiEmployee: number;
    esiEmployer: number;
    tds: number;
    pt: number;
    gratuity: number;
  };
  
  complianceByType: Array<{
    type: ComplianceRule['type'];
    total: number;
    amount: number;
  }>;
}

// Validation Types
export interface PolicyValidationResult {
  isValid: boolean;
  errors: string[];
  warnings: string[];
}

export interface ComplianceValidationResult {
  isValid: boolean;
  errors: string[];
  warnings: string[];
  calculatedAmounts?: {
    pfEmployee: number;
    pfEmployer: number;
    esiEmployee: number;
    esiEmployer: number;
    tds: number;
    pt: number;
    gratuity: number;
  };
}