import { PayrollRecord } from '../index';

// Payroll service specific types
export interface PayrollFilter {
  employeeId?: string;
  department?: string;
  status?: PayrollRecord['status'];
  month?: string;
  year?: number;
  salaryRange?: {
    min: number;
    max: number;
  };
  payDateRange?: {
    startDate: string;
    endDate: string;
  };
}

export interface PayrollSortOptions {
  field: 'employeeId' | 'month' | 'year' | 'basicSalary' | 'netSalary' | 'payDate' | 'status';
  order: 'asc' | 'desc';
}

export interface PayrollPaginationOptions {
  page: number;
  limit: number;
  sort?: PayrollSortOptions;
}

export interface PayrollListResponse {
  payrolls: PayrollRecord[];
  total: number;
  page: number;
  limit: number;
  totalPages: number;
}

export interface PayrollCreateRequest extends Omit<PayrollRecord, 'id' | 'netSalary' | 'payDate'> {}

export interface PayrollUpdateRequest extends Partial<Omit<PayrollRecord, 'id'>> {}

export interface PayrollBatchProcessRequest {
  month: string;
  year: number;
  employeeIds?: string[]; // If not provided, process all employees
  overwriteExisting?: boolean;
}

export interface PayrollStats {
  totalRecords: number;
  draftRecords: number;
  processedRecords: number;
  paidRecords: number;
  totalGrossPay: number;
  totalNetPay: number;
  totalDeductions: number;
  totalAllowances: number;
  totalOvertime: number;
  avgSalary: number;
  byDepartment: Array<{
    department: string;
    employeeCount: number;
    totalGross: number;
    totalNet: number;
    avgSalary: number;
  }>;
}

export interface PayrollComponent {
  id: string;
  name: string;
  type: 'earning' | 'deduction' | 'contribution';
  category: 'basic' | 'allowance' | 'tax' | 'insurance' | 'loan' | 'other';
  calculationType: 'fixed' | 'percentage' | 'formula';
  value?: number; // For fixed amounts
  percentage?: number; // For percentage-based
  formula?: string; // For complex calculations
  isActive: boolean;
  applicableRoles: string[];
  isMandatory: boolean;
  displayOrder: number;
}

export interface PayrollCalculation {
  employeeId: string;
  basicSalary: number;
  components: Array<{
    componentId: string;
    componentName: string;
    type: PayrollComponent['type'];
    amount: number;
    calculationDetails?: string;
  }>;
  grossPay: number;
  totalEarnings: number;
  totalDeductions: number;
  taxableIncome: number;
  taxAmount: number;
  netPay: number;
  overtimePay: number;
  calculatedAt: string;
}

export interface PayrollSlip {
  payrollId: string;
  employeeId: string;
  employeeName: string;
  employeeNumber: string;
  department: string;
  position: string;
  payPeriod: {
    month: string;
    year: number;
  };
  earnings: Array<{
    description: string;
    amount: number;
  }>;
  deductions: Array<{
    description: string;
    amount: number;
  }>;
  summary: {
    grossPay: number;
    totalDeductions: number;
    netPay: number;
    ytdGross: number;
    ytdNet: number;
  };
  paymentDetails: {
    payDate: string;
    paymentMethod: 'bank_transfer' | 'check' | 'cash';
    bankAccount?: string;
  };
  generatedAt: string;
}

export interface PayrollTax {
  id: string;
  name: string;
  type: 'income_tax' | 'social_security' | 'medicare' | 'state_tax' | 'local_tax';
  rate: number; // Percentage
  brackets?: Array<{
    min: number;
    max: number;
    rate: number;
  }>;
  exemptionAmount: number;
  isActive: boolean;
  applicableStates?: string[];
}

export interface PayrollReport {
  reportType: 'summary' | 'detailed' | 'tax' | 'department' | 'cost_center';
  period: {
    month?: string;
    year: number;
    quarter?: number;
  };
  filters?: PayrollFilter;
  data: {
    summary?: PayrollStats;
    records?: PayrollRecord[];
    taxSummary?: Array<{
      taxType: string;
      totalTax: number;
      employeeCount: number;
    }>;
    departmentBreakdown?: Array<{
      department: string;
      employeeCount: number;
      totalCost: number;
      avgCost: number;
    }>;
  };
  generatedAt: string;
  generatedBy: string;
}

export interface PayrollAuditLog {
  id: string;
  payrollId: string;
  action: 'created' | 'updated' | 'processed' | 'paid' | 'deleted' | 'recalculated';
  performedBy: string;
  performedAt: string;
  changes?: Array<{
    field: string;
    oldValue: any;
    newValue: any;
  }>;
  reason?: string;
}

export interface PayrollConfiguration {
  companyInfo: {
    name: string;
    address: string;
    taxId: string;
    payrollFrequency: 'weekly' | 'bi-weekly' | 'monthly' | 'semi-monthly';
  };
  paymentSettings: {
    defaultPaymentMethod: 'bank_transfer' | 'check' | 'cash';
    paymentCutoffDay: number; // Day of month
    processingDays: number; // Days before pay date to process
  };
  taxSettings: {
    defaultTaxYear: number;
    activeTaxes: string[]; // Tax component IDs
    taxFilingFrequency: 'monthly' | 'quarterly' | 'annually';
  };
  notifications: {
    sendPayslips: boolean;
    notifyProcessing: boolean;
    notifyCompletion: boolean;
    reminderDays: number;
  };
}

export interface PayrollApproval {
  id: string;
  month: string;
  year: number;
  employeeIds: string[];
  approvedBy: string;
  approvedAt: string;
  status: 'pending' | 'approved' | 'rejected';
  comments?: string;
  reviewedRecords: number;
  totalAmount: number;
}

export interface PayrollReconciliation {
  id: string;
  period: {
    month: string;
    year: number;
  };
  bankStatement: {
    totalPaid: number;
    transactionCount: number;
    failedTransactions: number;
  };
  payrollData: {
    totalCalculated: number;
    recordCount: number;
    processedCount: number;
  };
  discrepancies: Array<{
    type: 'amount_mismatch' | 'missing_payment' | 'duplicate_payment';
    employeeId: string;
    description: string;
    calculatedAmount: number;
    actualAmount?: number;
  }>;
  isReconciled: boolean;
  reconciledBy?: string;
  reconciledAt?: string;
}