// Reports service specific types
export interface ReportFilter {
  dateRange?: {
    startDate: string;
    endDate: string;
  };
  employeeIds?: string[];
  departmentIds?: string[];
  includeInactive?: boolean;
}

export interface ReportPaginationOptions {
  page: number;
  limit: number;
}

export interface ReportExportOptions {
  format: 'csv' | 'excel' | 'pdf';
  filename?: string;
  includeCharts?: boolean;
  includeRawData?: boolean;
}

export interface ReportExportResult {
  success: boolean;
  downloadUrl?: string;
  filename: string;
  format: string;
  error?: string;
}

// Employee Reports
export interface EmployeeReportData {
  totalEmployees: number;
  activeEmployees: number;
  inactiveEmployees: number;
  newHires: number;
  terminations: number;
  growthRate: number;
  departmentBreakdown: Array<{
    department: string;
    count: number;
    percentage: number;
    avgSalary: number;
  }>;
  positionBreakdown: Array<{
    position: string;
    count: number;
    avgSalary: number;
  }>;
  salaryAnalysis: {
    min: number;
    max: number;
    average: number;
    median: number;
    totalCost: number;
  };
  demographicBreakdown: {
    ageGroups: Array<{
      range: string;
      count: number;
    }>;
    genderDistribution: Array<{
      gender: string;
      count: number;
    }>;
    tenureDistribution: Array<{
      range: string;
      count: number;
    }>;
  };
}

export interface EmployeeGrowthTrend {
  month: string;
  employees: number;
  hires: number;
  terminations: number;
  netChange: number;
}

// Attendance Reports
export interface AttendanceReportData {
  totalRecords: number;
  attendanceRate: number;
  punctualityRate: number;
  avgWorkingHours: number;
  totalOvertimeHours: number;
  departmentStats: Array<{
    department: string;
    totalEmployees: number;
    present: number;
    absent: number;
    late: number;
    attendanceRate: number;
    avgWorkingHours: number;
  }>;
  dailyTrends: Array<{
    date: string;
    present: number;
    absent: number;
    late: number;
    rate: number;
  }>;
  weeklyPattern: Array<{
    dayOfWeek: string;
    avgAttendance: number;
    avgLateArrivals: number;
  }>;
}

export interface AttendanceInsights {
  topPerformers: Array<{
    employeeId: string;
    employeeName: string;
    attendanceRate: number;
    punctualityRate: number;
  }>;
  concernedEmployees: Array<{
    employeeId: string;
    employeeName: string;
    attendanceRate: number;
    issueCount: number;
    primaryIssue: string;
  }>;
  departmentComparison: Array<{
    department: string;
    rank: number;
    attendanceRate: number;
    improvement: number; // vs previous period
  }>;
}

// Leave Reports
export interface LeaveReportData {
  totalRequests: number;
  approvedRequests: number;
  rejectedRequests: number;
  pendingRequests: number;
  totalLeavesDays: number;
  leaveUtilizationRate: number;
  byType: Array<{
    type: string;
    count: number;
    totalDays: number;
    avgDuration: number;
  }>;
  byDepartment: Array<{
    department: string;
    totalRequests: number;
    approvedDays: number;
    utilizationRate: number;
  }>;
  monthlyTrend: Array<{
    month: string;
    requests: number;
    approvedDays: number;
  }>;
  seasonalPatterns: Array<{
    month: string;
    avgRequests: number;
    popularTypes: string[];
  }>;
}

export interface LeaveBalanceReport {
  employees: Array<{
    employeeId: string;
    employeeName: string;
    department: string;
    leaveBalances: Array<{
      type: string;
      allocated: number;
      used: number;
      remaining: number;
      utilization: number;
    }>;
  }>;
  departmentSummary: Array<{
    department: string;
    employeeCount: number;
    avgUtilization: number;
    totalUnusedDays: number;
  }>;
}

// Payroll Reports
export interface PayrollReportData {
  totalPayroll: number;
  totalEmployees: number;
  avgSalary: number;
  totalDeductions: number;
  totalAllowances: number;
  totalOvertime: number;
  byDepartment: Array<{
    department: string;
    employeeCount: number;
    totalCost: number;
    avgSalary: number;
    costPercentage: number;
  }>;
  monthlyTrend: Array<{
    month: string;
    totalPayroll: number;
    employeeCount: number;
    avgSalary: number;
  }>;
  salaryDistribution: Array<{
    range: string;
    count: number;
    percentage: number;
  }>;
}

export interface PayrollTaxReport {
  totalTaxAmount: number;
  taxBreakdown: Array<{
    taxType: string;
    amount: number;
    percentage: number;
    employeeCount: number;
  }>;
  departmentTaxes: Array<{
    department: string;
    totalTax: number;
    avgTaxPerEmployee: number;
  }>;
  quarterlyComparison: Array<{
    quarter: string;
    totalTax: number;
    change: number;
  }>;
}

// Department Reports
export interface DepartmentReportData {
  totalDepartments: number;
  totalEmployees: number;
  totalBudget: number;
  avgDepartmentSize: number;
  departments: Array<{
    id: string;
    name: string;
    employeeCount: number;
    budget: number;
    costPerEmployee: number;
    budgetUtilization: number;
    performanceMetrics: {
      attendanceRate: number;
      leaveUtilization: number;
      turnoverRate: number;
    };
  }>;
  budgetAnalysis: {
    totalAllocated: number;
    totalSpent: number;
    utilizationRate: number;
    departmentRanking: Array<{
      department: string;
      utilization: number;
      efficiency: number;
    }>;
  };
}

// Performance Reports
export interface PerformanceReportData {
  overallRating: number;
  totalReviews: number;
  reviewCompletion: number;
  ratingDistribution: Array<{
    rating: string;
    count: number;
    percentage: number;
  }>;
  departmentPerformance: Array<{
    department: string;
    avgRating: number;
    completionRate: number;
    improvementAreas: string[];
  }>;
  topPerformers: Array<{
    employeeId: string;
    employeeName: string;
    rating: number;
    department: string;
  }>;
  goalCompletion: Array<{
    goalCategory: string;
    totalGoals: number;
    completedGoals: number;
    completionRate: number;
  }>;
}

// Report Templates
export interface ReportTemplate {
  id: string;
  name: string;
  description: string;
  type: 'employee' | 'attendance' | 'leave' | 'payroll' | 'department' | 'performance';
  filters: ReportFilter;
  columns: string[];
  groupBy?: string[];
  sortBy?: string;
  isPublic: boolean;
  createdBy: string;
  createdAt: string;
  lastUsed?: string;
  usageCount: number;
}

// Scheduled Reports
export interface ScheduledReport {
  id: string;
  name: string;
  templateId: string;
  schedule: {
    frequency: 'daily' | 'weekly' | 'monthly' | 'quarterly';
    dayOfWeek?: number; // 0-6 for weekly
    dayOfMonth?: number; // 1-31 for monthly
    time: string; // HH:MM
  };
  recipients: Array<{
    email: string;
    name: string;
    type: 'to' | 'cc' | 'bcc';
  }>;
  format: ReportExportOptions['format'];
  isActive: boolean;
  lastRun?: string;
  nextRun: string;
  createdBy: string;
}

// Report Analytics
export interface ReportAnalytics {
  reportId: string;
  reportType: string;
  usage: {
    totalViews: number;
    totalDownloads: number;
    avgViewDuration: number;
    popularFormats: Array<{
      format: string;
      count: number;
    }>;
  };
  performance: {
    avgGenerationTime: number;
    cacheHitRate: number;
    errorRate: number;
  };
  trends: Array<{
    date: string;
    views: number;
    downloads: number;
  }>;
}

// Custom Report Builder
export interface CustomReportConfig {
  id: string;
  name: string;
  description: string;
  dataSource: 'employees' | 'attendance' | 'leaves' | 'payroll' | 'departments';
  fields: Array<{
    fieldName: string;
    displayName: string;
    dataType: 'string' | 'number' | 'date' | 'boolean';
    aggregation?: 'sum' | 'avg' | 'count' | 'min' | 'max';
    isVisible: boolean;
    order: number;
  }>;
  filters: Array<{
    field: string;
    operator: 'equals' | 'not_equals' | 'contains' | 'greater_than' | 'less_than' | 'between';
    value: any;
  }>;
  groupBy: string[];
  sortBy: Array<{
    field: string;
    direction: 'asc' | 'desc';
  }>;
  createdBy: string;
  createdAt: string;
  isShared: boolean;
}

export interface ReportData {
  employee?: EmployeeReportData;
  attendance?: AttendanceReportData;
  leave?: LeaveReportData;
  payroll?: PayrollReportData;
  department?: DepartmentReportData;
  performance?: PerformanceReportData;
}

export interface GeneratedReport {
  id: string;
  name: string;
  type: string;
  data: ReportData;
  filters: ReportFilter;
  generatedAt: string;
  generatedBy: string;
  expiresAt?: string;
  downloadCount: number;
}

export interface ReportValidation {
  isValid: boolean;
  errors: Array<{
    field: string;
    message: string;
  }>;
  warnings: Array<{
    field: string;
    message: string;
  }>;
}