// Comprehensive form validation utilities
import { useState } from 'react';

export interface ValidationRule {
  required?: boolean;
  minLength?: number;
  maxLength?: number;
  min?: number;
  max?: number;
  pattern?: RegExp;
  customValidator?: (value: any) => boolean;
  errorMessage?: string;
}

export interface ValidationResult {
  isValid: boolean;
  errors: Record<string, string>;
}

export interface FieldValidation {
  [fieldName: string]: ValidationRule[];
}

// Common validation patterns
export const VALIDATION_PATTERNS = {
  EMAIL: /^[^\s@]+@[^\s@]+\.[^\s@]+$/,
  PHONE: /^[+]?[()]?[\d\s\-()]{10,}$/,
  EMPLOYEE_ID: /^EMP\d{3,6}$/,
  POSTAL_CODE: /^\d{5}(-\d{4})?$/,
  PASSWORD: /^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]{8,}$/,
  NAME: /^[a-zA-Z\s\-']+$/,
  DECIMAL: /^\d+(\.\d{1,2})?$/,
  TIME: /^([0-1]?[0-9]|2[0-3]):[0-5][0-9]$/,
  DATE: /^\d{4}-\d{2}-\d{2}$/,
} as const;

// Common validation messages
export const VALIDATION_MESSAGES = {
  REQUIRED: 'This field is required',
  EMAIL: 'Please enter a valid email address',
  PHONE: 'Please enter a valid phone number',
  MIN_LENGTH: (min: number) => `Must be at least ${min} characters long`,
  MAX_LENGTH: (max: number) => `Must not exceed ${max} characters`,
  MIN_VALUE: (min: number) => `Must be at least ${min}`,
  MAX_VALUE: (max: number) => `Must not exceed ${max}`,
  PATTERN: 'Invalid format',
  PASSWORD: 'Password must contain at least 8 characters with uppercase, lowercase, number and special character',
  EMPLOYEE_ID: 'Employee ID must be in format EMP123456',
  NAME: 'Name can only contain letters, spaces, hyphens and apostrophes',
  DECIMAL: 'Please enter a valid decimal number',
  TIME: 'Please enter time in HH:MM format',
  DATE: 'Please enter date in YYYY-MM-DD format',
  DATE_RANGE: 'End date must be after start date',
  FUTURE_DATE: 'Date cannot be in the future',
  PAST_DATE: 'Date cannot be in the past',
  WORKING_HOURS: 'Working hours cannot exceed 24 hours',
  SALARY_RANGE: 'Salary must be between $0 and $999,999',
} as const;

// Basic validation functions
export class FormValidator {
  static validateField(value: any, rules: ValidationRule[]): string[] {
    const errors: string[] = [];

    for (const rule of rules) {
      if (rule.required && this.isEmpty(value)) {
        errors.push(rule.errorMessage || VALIDATION_MESSAGES.REQUIRED);
        continue;
      }

      if (this.isEmpty(value)) continue; // Skip other validations if value is empty and not required

      if (rule.minLength && String(value).length < rule.minLength) {
        errors.push(rule.errorMessage || VALIDATION_MESSAGES.MIN_LENGTH(rule.minLength));
      }

      if (rule.maxLength && String(value).length > rule.maxLength) {
        errors.push(rule.errorMessage || VALIDATION_MESSAGES.MAX_LENGTH(rule.maxLength));
      }

      if (rule.min && Number(value) < rule.min) {
        errors.push(rule.errorMessage || VALIDATION_MESSAGES.MIN_VALUE(rule.min));
      }

      if (rule.max && Number(value) > rule.max) {
        errors.push(rule.errorMessage || VALIDATION_MESSAGES.MAX_VALUE(rule.max));
      }

      if (rule.pattern && !rule.pattern.test(String(value))) {
        errors.push(rule.errorMessage || VALIDATION_MESSAGES.PATTERN);
      }

      if (rule.customValidator && !rule.customValidator(value)) {
        errors.push(rule.errorMessage || 'Invalid value');
      }
    }

    return errors;
  }

  static validateForm(data: Record<string, any>, validationSchema: FieldValidation): ValidationResult {
    const errors: Record<string, string> = {};
    let isValid = true;

    for (const [fieldName, rules] of Object.entries(validationSchema)) {
      const fieldErrors = this.validateField(data[fieldName], rules);
      if (fieldErrors.length > 0) {
        errors[fieldName] = fieldErrors[0]; // Take first error
        isValid = false;
      }
    }

    return { isValid, errors };
  }

  private static isEmpty(value: any): boolean {
    return value === null || value === undefined || value === '' || 
           (Array.isArray(value) && value.length === 0);
  }

  // Specific field validators
  static isValidEmail(email: string): boolean {
    return VALIDATION_PATTERNS.EMAIL.test(email);
  }

  static isValidPhone(phone: string): boolean {
    return VALIDATION_PATTERNS.PHONE.test(phone);
  }

  static isValidEmployeeId(employeeId: string): boolean {
    return VALIDATION_PATTERNS.EMPLOYEE_ID.test(employeeId);
  }

  static isValidPassword(password: string): boolean {
    return VALIDATION_PATTERNS.PASSWORD.test(password);
  }

  static isValidDate(date: string): boolean {
    const parsedDate = new Date(date);
    return !isNaN(parsedDate.getTime()) && VALIDATION_PATTERNS.DATE.test(date);
  }

  static isValidTime(time: string): boolean {
    return VALIDATION_PATTERNS.TIME.test(time);
  }

  static isValidDateRange(startDate: string, endDate: string): boolean {
    if (!this.isValidDate(startDate) || !this.isValidDate(endDate)) {
      return false;
    }
    return new Date(startDate) < new Date(endDate);
  }

  static isValidSalary(salary: number): boolean {
    return salary >= 0 && salary <= 999999;
  }

  static isValidWorkingHours(hours: number): boolean {
    return hours >= 0 && hours <= 24;
  }

  // Advanced validators
  static isValidAge(birthDate: string): boolean {
    const birth = new Date(birthDate);
    const today = new Date();
    const age = today.getFullYear() - birth.getFullYear();
    return age >= 16 && age <= 100;
  }

  static isValidFutureDate(date: string): boolean {
    return new Date(date) > new Date();
  }

  static isValidPastDate(date: string): boolean {
    return new Date(date) < new Date();
  }

  static isUniqueEmail(email: string, existingEmails: string[]): boolean {
    return !existingEmails.includes(email.toLowerCase());
  }

  static isUniqueEmployeeId(employeeId: string, existingIds: string[]): boolean {
    return !existingIds.includes(employeeId.toUpperCase());
  }
}

// Pre-defined validation schemas
export const VALIDATION_SCHEMAS = {
  EMPLOYEE: {
    firstName: [
      { required: true, errorMessage: 'First name is required' },
      { minLength: 2, errorMessage: 'First name must be at least 2 characters' },
      { maxLength: 50, errorMessage: 'First name cannot exceed 50 characters' },
      { pattern: VALIDATION_PATTERNS.NAME, errorMessage: VALIDATION_MESSAGES.NAME },
    ],
    lastName: [
      { required: true, errorMessage: 'Last name is required' },
      { minLength: 2, errorMessage: 'Last name must be at least 2 characters' },
      { maxLength: 50, errorMessage: 'Last name cannot exceed 50 characters' },
      { pattern: VALIDATION_PATTERNS.NAME, errorMessage: VALIDATION_MESSAGES.NAME },
    ],
    email: [
      { required: true, errorMessage: 'Email is required' },
      { pattern: VALIDATION_PATTERNS.EMAIL, errorMessage: VALIDATION_MESSAGES.EMAIL },
      { maxLength: 100, errorMessage: 'Email cannot exceed 100 characters' },
    ],
    employeeId: [
      { required: true, errorMessage: 'Employee ID is required' },
      { pattern: VALIDATION_PATTERNS.EMPLOYEE_ID, errorMessage: VALIDATION_MESSAGES.EMPLOYEE_ID },
    ],
    phone: [
      { pattern: VALIDATION_PATTERNS.PHONE, errorMessage: VALIDATION_MESSAGES.PHONE },
    ],
    department: [
      { required: true, errorMessage: 'Department is required' },
      { minLength: 2, errorMessage: 'Department must be at least 2 characters' },
    ],
    position: [
      { required: true, errorMessage: 'Position is required' },
      { minLength: 2, errorMessage: 'Position must be at least 2 characters' },
      { maxLength: 100, errorMessage: 'Position cannot exceed 100 characters' },
    ],
    salary: [
      { required: true, errorMessage: 'Salary is required' },
      { min: 0, errorMessage: 'Salary cannot be negative' },
      { max: 999999, errorMessage: VALIDATION_MESSAGES.SALARY_RANGE },
    ],
    hireDate: [
      { required: true, errorMessage: 'Hire date is required' },
      { pattern: VALIDATION_PATTERNS.DATE, errorMessage: VALIDATION_MESSAGES.DATE },
    ],
    dateOfBirth: [
      { pattern: VALIDATION_PATTERNS.DATE, errorMessage: VALIDATION_MESSAGES.DATE },
      { customValidator: (value: any) => !value || FormValidator.isValidAge(value), errorMessage: 'Age must be between 16 and 100 years' },
    ],
    address: [
      { maxLength: 200, errorMessage: 'Address cannot exceed 200 characters' },
    ],
  },

  DEPARTMENT: {
    name: [
      { required: true, errorMessage: 'Department name is required' },
      { minLength: 2, errorMessage: 'Department name must be at least 2 characters' },
      { maxLength: 100, errorMessage: 'Department name cannot exceed 100 characters' },
    ],
    description: [
      { maxLength: 500, errorMessage: 'Description cannot exceed 500 characters' },
    ],
    managerId: [
      { pattern: VALIDATION_PATTERNS.EMPLOYEE_ID, errorMessage: 'Manager ID must be a valid employee ID' },
    ],
    budget: [
      { required: true, errorMessage: 'Budget is required' },
      { min: 0, errorMessage: 'Budget cannot be negative' },
      { max: 99999999, errorMessage: 'Budget cannot exceed $99,999,999' },
    ],
  },

  ATTENDANCE: {
    employeeId: [
      { required: true, errorMessage: 'Employee ID is required' },
      { pattern: VALIDATION_PATTERNS.EMPLOYEE_ID, errorMessage: VALIDATION_MESSAGES.EMPLOYEE_ID },
    ],
    date: [
      { required: true, errorMessage: 'Date is required' },
      { pattern: VALIDATION_PATTERNS.DATE, errorMessage: VALIDATION_MESSAGES.DATE },
    ],
    checkIn: [
      { pattern: VALIDATION_PATTERNS.TIME, errorMessage: VALIDATION_MESSAGES.TIME },
    ],
    checkOut: [
      { pattern: VALIDATION_PATTERNS.TIME, errorMessage: VALIDATION_MESSAGES.TIME },
    ],
    workingHours: [
      { min: 0, errorMessage: 'Working hours cannot be negative' },
      { max: 24, errorMessage: VALIDATION_MESSAGES.WORKING_HOURS },
    ],
    overtime: [
      { min: 0, errorMessage: 'Overtime cannot be negative' },
      { max: 12, errorMessage: 'Overtime cannot exceed 12 hours' },
    ],
    notes: [
      { maxLength: 500, errorMessage: 'Notes cannot exceed 500 characters' },
    ],
  },

  LEAVE: {
    employeeId: [
      { required: true, errorMessage: 'Employee ID is required' },
      { pattern: VALIDATION_PATTERNS.EMPLOYEE_ID, errorMessage: VALIDATION_MESSAGES.EMPLOYEE_ID },
    ],
    type: [
      { required: true, errorMessage: 'Leave type is required' },
    ],
    startDate: [
      { required: true, errorMessage: 'Start date is required' },
      { pattern: VALIDATION_PATTERNS.DATE, errorMessage: VALIDATION_MESSAGES.DATE },
    ],
    endDate: [
      { required: true, errorMessage: 'End date is required' },
      { pattern: VALIDATION_PATTERNS.DATE, errorMessage: VALIDATION_MESSAGES.DATE },
    ],
    reason: [
      { required: true, errorMessage: 'Reason is required' },
      { minLength: 10, errorMessage: 'Reason must be at least 10 characters' },
      { maxLength: 500, errorMessage: 'Reason cannot exceed 500 characters' },
    ],
  },

  PAYROLL: {
    employeeId: [
      { required: true, errorMessage: 'Employee ID is required' },
      { pattern: VALIDATION_PATTERNS.EMPLOYEE_ID, errorMessage: VALIDATION_MESSAGES.EMPLOYEE_ID },
    ],
    month: [
      { required: true, errorMessage: 'Month is required' },
    ],
    year: [
      { required: true, errorMessage: 'Year is required' },
      { min: 2000, errorMessage: 'Year must be from 2000 onwards' },
      { max: new Date().getFullYear() + 1, errorMessage: 'Year cannot be more than next year' },
    ],
    basicSalary: [
      { required: true, errorMessage: 'Basic salary is required' },
      { min: 0, errorMessage: 'Basic salary cannot be negative' },
      { max: 999999, errorMessage: 'Basic salary cannot exceed $999,999' },
    ],
    allowances: [
      { min: 0, errorMessage: 'Allowances cannot be negative' },
      { max: 99999, errorMessage: 'Allowances cannot exceed $99,999' },
    ],
    deductions: [
      { min: 0, errorMessage: 'Deductions cannot be negative' },
      { max: 99999, errorMessage: 'Deductions cannot exceed $99,999' },
    ],
    overtime: [
      { min: 0, errorMessage: 'Overtime cannot be negative' },
      { max: 99999, errorMessage: 'Overtime cannot exceed $99,999' },
    ],
  },

  USER: {
    email: [
      { required: true, errorMessage: 'Email is required' },
      { pattern: VALIDATION_PATTERNS.EMAIL, errorMessage: VALIDATION_MESSAGES.EMAIL },
    ],
    password: [
      { required: true, errorMessage: 'Password is required' },
      { minLength: 8, errorMessage: 'Password must be at least 8 characters' },
      { pattern: VALIDATION_PATTERNS.PASSWORD, errorMessage: VALIDATION_MESSAGES.PASSWORD },
    ],
    name: [
      { required: true, errorMessage: 'Name is required' },
      { minLength: 2, errorMessage: 'Name must be at least 2 characters' },
      { maxLength: 100, errorMessage: 'Name cannot exceed 100 characters' },
      { pattern: VALIDATION_PATTERNS.NAME, errorMessage: VALIDATION_MESSAGES.NAME },
    ],
    role: [
      { required: true, errorMessage: 'Role is required' },
    ],
  },

  LOGIN: {
    email: [
      { required: true, errorMessage: 'Email is required' },
      { pattern: VALIDATION_PATTERNS.EMAIL, errorMessage: VALIDATION_MESSAGES.EMAIL },
    ],
    password: [
      { required: true, errorMessage: 'Password is required' },
      { minLength: 6, errorMessage: 'Password must be at least 6 characters' },
    ],
  },
};

// React Hook for form validation
export const useFormValidation = (validationSchema: FieldValidation) => {
  const [errors, setErrors] = useState<Record<string, string>>({});
  const [touched, setTouched] = useState<Record<string, boolean>>({});

  const validateField = (fieldName: string, value: any) => {
    const fieldRules = validationSchema[fieldName];
    if (!fieldRules) return [];

    return FormValidator.validateField(value, fieldRules);
  };

  const validateForm = (data: Record<string, any>) => {
    const result = FormValidator.validateForm(data, validationSchema);
    setErrors(result.errors);
    return result;
  };

  const handleFieldChange = (fieldName: string, value: any) => {
    setTouched(prev => ({ ...prev, [fieldName]: true }));
    
    const fieldErrors = validateField(fieldName, value);
    setErrors(prev => ({
      ...prev,
      [fieldName]: fieldErrors.length > 0 ? fieldErrors[0] : '',
    }));
  };

  const clearErrors = () => {
    setErrors({});
    setTouched({});
  };

  const hasError = (fieldName: string) => {
    return touched[fieldName] && !!errors[fieldName];
  };

  const getErrorMessage = (fieldName: string) => {
    return hasError(fieldName) ? errors[fieldName] : '';
  };

  return {
    errors,
    touched,
    validateField,
    validateForm,
    handleFieldChange,
    clearErrors,
    hasError,
    getErrorMessage,
  };
};